package com.digiwin.dap.middle.ram.filter;

import com.digiwin.dap.middle.ram.domain.request.AccessResult;
import com.digiwin.dap.middle.ram.domain.request.AccessUser;
import com.digiwin.dap.middle.ram.support.security.auth.AuthPolicyHandler;
import com.digiwin.dap.middle.ram.support.security.path.PathPolicyHandler;
import com.digiwin.dap.middleware.auth.AppAuthContext;
import com.digiwin.dap.middleware.auth.AppAuthContextHolder;
import com.digiwin.dap.middleware.auth.domain.AuthResult;
import com.digiwin.dap.middleware.auth.domain.AuthType;
import com.digiwin.dap.middleware.auth.domain.RamVersion;
import com.digiwin.dap.middleware.auth.domain.RequestInfo;
import com.digiwin.dap.middleware.constant.InternalUrl;
import com.digiwin.dap.middleware.domain.FilterOrderEnum;
import org.springframework.core.Ordered;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * 访问控制RAM（Resource Access Management）<br>
 * 管理用户身份与资源访问权限
 *
 * @author fobgochod
 * @see RamVersion#v1
 */
public class RamV1AccessCheckFilter extends OncePerRequestFilter implements Ordered {

    private final PathPolicyHandler<AccessUser> pathPolicyHandler;
    private final AuthPolicyHandler authPolicyHandler;

    public RamV1AccessCheckFilter(PathPolicyHandler<AccessUser> pathPolicyHandler, AuthPolicyHandler authPolicyHandler) {
        this.pathPolicyHandler = pathPolicyHandler;
        this.authPolicyHandler = authPolicyHandler;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        String requestURI = request.getRequestURI();
        final AppAuthContext context = AppAuthContextHolder.getContext();
        RequestInfo requestInfo = context.getRequestInfo();
        if (InternalUrl.tokenAnalyzes().containsKey(requestURI)) {
            accessCheck(context, requestInfo);
        } else if (InternalUrl.authAllows().containsKey(requestURI) || context.isUpstream()) {
            /**
             * 1.token解析不需要接口鉴权、但是需要授权检查，这地方不放，在策略第一步放行
             * 2.登录不需要接口鉴权，也不需要在拦截器授权检查，延时在登录逻辑检查
             * 3.中间就内部调用接口，都不检查0
             * 4.鉴权需要的相关接口，不检查防止陷入死循环（ps正常应该都是3内部调用）
             */
        } else {
            accessCheck(context, requestInfo);
        }
        filterChain.doFilter(request, response);
    }

    private void accessCheck(final AppAuthContext context, RequestInfo requestInfo) {
        AccessUser requestUser = AccessUser.of(requestInfo.getAppId(), requestInfo.getMethod(), requestInfo.getPath());
        // 接口访问控制
        AccessResult accessResult = pathPolicyHandler.handle(requestUser);
        context.setResultType(accessResult.getResultType().name());
        accessResult.checkAccessResult();
        // 授权访问控制
        AuthResult authResult = authPolicyHandler.process(AuthResult.of(), requestUser);
        AuthType.checkAuthResult(authResult.getAuthType(), requestUser.getUserId(), requestUser.getTenantId(), requestUser.getSysId());
    }

    @Override
    public int getOrder() {
        return FilterOrderEnum.ACCESS_CHECK.order();
    }
}
