package com.digiwin.dap.middle.ram.support;

import com.digiwin.dap.middle.ram.constant.RamI18nError;
import com.digiwin.dap.middle.ram.domain.remote.AppAuthResult;
import com.digiwin.dap.middle.ram.domain.remote.Function;
import com.digiwin.dap.middle.ram.domain.request.AccessUser;
import com.digiwin.dap.middleware.auth.domain.AuthResult;
import com.digiwin.dap.middleware.constant.BeanConstants;
import com.digiwin.dap.middleware.constant.GlobalConstants;
import com.digiwin.dap.middleware.constant.InternalUrl;
import com.digiwin.dap.middleware.domain.CommonErrorCode;
import com.digiwin.dap.middleware.domain.DapEnv;
import com.digiwin.dap.middleware.domain.StdData;
import com.digiwin.dap.middleware.exception.ThirdCallException;
import com.digiwin.dap.middleware.exception.UnauthorizedException;
import com.digiwin.dap.middleware.util.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 默认实现
 *
 * @author fobgochod
 */
public class DefaultRamHandlerSupport implements RamHandlerSupport {

    private static final Logger logger = LoggerFactory.getLogger(DefaultRamHandlerSupport.class);
    private static final String USER_WHITE_LIST = "integration;virtual";
    @Autowired
    private DapEnv dapEnv;
    @Resource(name = BeanConstants.DAP_RETRY_TEMPLATE)
    private RestTemplate retryRestTemplate;

    @Override
    public AuthResult getAuthResult(AuthResult authResult, AccessUser request) {
        AppAuthResult result = getAuthResult(request.getTenantId(), request.getUserId(),
                request.getSysId(), request.isPlatform(), authResult.getTenants(),
                !USER_WHITE_LIST.contains(request.getUserId()));
        return result.flushAuthType();
    }

    @Override
    public List<String> getUserFunction(AccessUser request) {
        return getUserFunction(request.getUserId(), request.getTenantId(), request.getSysId(), request.getUserToken())
                .uris(request.getSysId());
    }

    private AppAuthResult getAuthResult(String tenantId, String userId,
                                        String appId, boolean platform,
                                        List<String> tenants, boolean checkUserAuth) {
        String url = dapEnv.getCacUri() + InternalUrl.AUTH_CHECK;
        Map<String, Object> body = new HashMap<>();
        try {
            body.put("tenantId", tenantId);
            body.put("userId", userId);
            body.put("goodsCode", appId);
            body.put("tenants", tenants);
            body.put("checkUserAuth", Boolean.toString(checkUserAuth));
            if (platform) {
                // 是平台，则查询旗下的应用
                body.put("apps", this.getAppIds(appId, true));
            } else {
                // 是应用，则查询该应用的平台
                body.put("platformApps", this.getAppIds(appId, false));
            }
            HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(body);
            return retryRestTemplate.postForObject(url, httpEntity, AppAuthResult.class);
        } catch (Exception e) {
            logger.error(ExceptionUtils.logError("获取用户应用授权", url, body, e), e);
            throw new UnauthorizedException(RamI18nError.CAC_AUTH_GET_FAIL, new String[]{tenantId, userId, appId}, 411005);
        }
    }

    private List<String> getAppIds(String appId, boolean platform) {
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            Map<String, Object> requestBody = new HashMap<>();
            requestBody.put("id", appId);
            requestBody.put("platform", platform);
            HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(requestBody, headers);

            String uri = dapEnv.getIamUri() + InternalUrl.SYS_PLATFORM_IDS;
            ResponseEntity<StdData<List<String>>> result = retryRestTemplate.exchange(uri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<StdData<List<String>>>() {
            });
            return result.getBody().getData();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return Collections.emptyList();
    }

    private Function getUserFunction(String userId, String tenantId, String sysId, String userToken) {
        String uri = dapEnv.getIamUri() + "/api/iam/v2/permission/user/all";
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            headers.add(GlobalConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);

            Map<String, String> body = new HashMap<>(3);
            body.put("tenantId", tenantId);
            body.put("userId", userId);
            body.put("target", "drn:iam:app:" + sysId);
            HttpEntity<Map<String, String>> request = new HttpEntity<>(body, headers);
            return retryRestTemplate.postForObject(uri, request, Function.class);
        } catch (Exception e) {
            logger.error("获取功能权限接口异常", e);
            throw new ThirdCallException(CommonErrorCode.REMOTE_UNEXPECTED, uri, e);
        }
    }
}
