package com.digiwin.dap.middle.ram.support.security.auth.handler;

import com.digiwin.dap.middle.ram.domain.request.AccessUser;
import com.digiwin.dap.middle.ram.domain.enums.PolicyType;
import com.digiwin.dap.middle.ram.domain.enums.TargetType;
import com.digiwin.dap.middle.ram.service.core.RamCoreService;
import com.digiwin.dap.middle.ram.support.security.auth.AuthPolicyHandler;
import com.digiwin.dap.middleware.auth.domain.AuthResult;
import com.digiwin.dap.middleware.auth.domain.AuthType;
import org.springframework.core.annotation.Order;

import java.util.List;

/**
 * 不做用户加入检查，即存在租户应用关系，但是未给用户授权该应用<br>
 * 策略类型 {@link PolicyType#NoAuthUser}<br>
 * <ol>
 *     <li>指定应用不检查用户授权</li>
 *     <li>指定应用租户不检查用户授权</li>
 * </ol>
 *
 * @author fobgochod
 */
@Order(20)
public class NoAuthUserAuthPolicyHandler extends AuthPolicyHandler {

    public NoAuthUserAuthPolicyHandler(RamCoreService ramCoreService) {
        this.ramCoreService = ramCoreService;
    }

    @Override
    public boolean supports(AuthType authType) {
        return authType == AuthType.AppNoneUser;
    }

    @Override
    public AuthResult process(AuthResult authResult, AccessUser request) {
        // 应用层级，该应用所有用户不用检查
        List<String> policyId = ramCoreService.findPolicyId(request.getApp(), PolicyType.NoAuthUser.name(), TargetType.Sys.name(), request.getSysId());
        if (!policyId.isEmpty()) {
            authResult.setAuthType(AuthType.AppOk);
        } else {
            // 租户-应用层级，该租户购买的应用用户不用授权检查
            policyId = ramCoreService.findPolicyId(request.getApp(), PolicyType.NoAuthUser.name(), TargetType.TenantSys.name(), request.getTenantId() + '-' + request.getSysId());
            if (!policyId.isEmpty()) {
                authResult.setAuthType(AuthType.AppOk);
            }
        }
        return authResult;
    }
}
