package com.digiwin.dap.middle.ram.support.web.condition;

import org.springframework.lang.Nullable;

import javax.servlet.DispatcherType;
import javax.servlet.http.HttpServletRequest;
import java.util.*;

/**
 * A logical disjunction (' || ') request condition that matches a request
 * against a set of {@link HttpMethod RequestMethods}.
 *
 * @author fobgochod
 */
public final class MethodsCondition extends AbstractCondition {

    private static final MethodsCondition GET_CONDITION = new MethodsCondition(HttpMethod.GET);

    private final Set<HttpMethod> methods;


    /**
     * Create a new instance with the given request methods.
     *
     * @param requestMethods 0 or more HTTP request methods;
     *                       if, 0 the condition will match to every request
     */
    public MethodsCondition(HttpMethod... requestMethods) {
        this(Arrays.asList(requestMethods));
    }

    private MethodsCondition(Collection<HttpMethod> requestMethods) {
        this.methods = Collections.unmodifiableSet(new LinkedHashSet<>(requestMethods));
    }

    /**
     * Returns all {@link HttpMethod RequestMethods} contained in this condition.
     */
    public Set<HttpMethod> getMethods() {
        return this.methods;
    }

    @Override
    protected Collection<HttpMethod> getContent() {
        return this.methods;
    }

    /**
     * Check if any of the HTTP request methods match the given request and
     * return an instance that contains the matching HTTP request method only.
     *
     * @param httpMethodValue the current method
     * @return the same instance if the condition is empty (unless the request
     * method is HTTP OPTIONS), a new condition with the matched request method,
     * or {@code null} if there is no match or the condition is empty and the
     * request method is OPTIONS.
     */
    @Nullable
    public MethodsCondition getMatchingCondition(String httpMethodValue) {
        HttpMethod method = HttpMethod.resolve(httpMethodValue);
        if (getMethods().isEmpty()) {
            if (HttpMethod.OPTIONS == method) {
                // No implicit match for OPTIONS (we handle it)
                return null;
            }
            return this;
        }

        return matchRequestMethod(method);
    }

    @Nullable
    public MethodsCondition getMatchingCondition(HttpServletRequest request) {
        HttpMethod method = HttpMethod.resolve(request.getMethod());
        if (getMethods().isEmpty()) {
            if ((HttpMethod.OPTIONS == method) && !DispatcherType.ERROR.equals(request.getDispatcherType())) {
                // No implicit match for OPTIONS (we handle it)
                return null;
            }
            return this;
        }

        return matchRequestMethod(method);
    }

    @Nullable
    private MethodsCondition matchRequestMethod(HttpMethod httpMethod) {
        if (httpMethod != null) {
            for (HttpMethod method : getMethods()) {
                if (httpMethod == method) {
                    return new MethodsCondition(method);
                }
            }
            if (httpMethod == HttpMethod.HEAD && getMethods().contains(HttpMethod.GET)) {
                return GET_CONDITION;
            }
        }
        return null;
    }
}
