package com.digiwin.dap.middle.ram.support.web.condition;

import com.digiwin.dap.middleware.commons.util.StrUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.PathMatcher;
import org.springframework.web.util.UrlPathHelper;

import javax.servlet.http.HttpServletRequest;
import java.util.*;

/**
 * A logical disjunction (' || ') request condition that matches a request
 * against a set of URL path patterns.
 *
 * @author fobgochod
 */
public final class PatternsCondition extends AbstractCondition {

    private final Set<String> patterns;
    private final UrlPathHelper pathHelper;
    private final PathMatcher pathMatcher;


    /**
     * Creates a new instance with the given URL patterns.
     * Each pattern that is not empty and does not start with "/" is prepended with "/".
     *
     * @param patterns 0 or more URL patterns; if 0 the condition will match to every request.
     */
    public PatternsCondition(String... patterns) {
        this(Arrays.asList(patterns), null, null);
    }

    /**
     * Additional constructor with flags for using suffix pattern (.*) and
     * trailing slash matches.
     *
     * @param patterns      the URL patterns to use; if 0, the condition will match to every request.
     * @param urlPathHelper for determining the lookup path of a request
     * @param pathMatcher   for path matching with patterns
     */
    public PatternsCondition(String[] patterns, @Nullable UrlPathHelper urlPathHelper, @Nullable PathMatcher pathMatcher) {
        this(Arrays.asList(patterns), urlPathHelper, pathMatcher);
    }

    /**
     * Private constructor accepting a collection of patterns.
     */
    private PatternsCondition(Collection<String> patterns, @Nullable UrlPathHelper urlPathHelper, @Nullable PathMatcher pathMatcher) {
        this.patterns = Collections.unmodifiableSet(prependLeadingSlash(patterns));
        this.pathHelper = (urlPathHelper != null ? urlPathHelper : MappingInfo.BuilderConfiguration.defaultInstance);
        this.pathMatcher = (pathMatcher != null ? pathMatcher : new AntPathMatcher());
    }


    private static Set<String> prependLeadingSlash(Collection<String> patterns) {
        Set<String> result = new LinkedHashSet<>(patterns.size());
        for (String pattern : patterns) {
            if (StrUtils.isNotEmpty(pattern) && !pattern.startsWith("/")) {
                pattern = "/" + pattern;
            }
            result.add(pattern);
        }
        return result;
    }

    public Set<String> getPatterns() {
        return this.patterns;
    }

    @Override
    protected Collection<String> getContent() {
        return this.patterns;
    }

    /**
     * Checks if any of the patterns match the given request and returns an instance
     * that is guaranteed to contain matching patterns, sorted via
     * {@link PathMatcher#getPatternComparator(String)}.
     * <p>A matching pattern is obtained by making checks in the following order:
     * <ul>
     * <li>Direct match
     * <li>Pattern match with ".*" appended if the pattern doesn't already contain a "."
     * <li>Pattern match
     * <li>Pattern match with "/" appended if the pattern doesn't already end in "/"
     * </ul>
     *
     * @param lookupPath the current request
     * @return the same instance if the condition contains no patterns;
     * or a new condition with sorted matching patterns;
     * or {@code null} if no patterns match.
     */
    public PatternsCondition getMatchingCondition(String lookupPath) {
        if (this.patterns.isEmpty()) {
            return this;
        }
        List<String> matches = getMatchingPatterns(lookupPath);
        return (!matches.isEmpty() ? new PatternsCondition(matches, this.pathHelper, this.pathMatcher) : null);
    }

    public PatternsCondition getMatchingCondition(HttpServletRequest request) {
        if (this.patterns.isEmpty()) {
            return this;
        }
        String lookupPath = this.pathHelper.getLookupPathForRequest(request);
        List<String> matches = getMatchingPatterns(lookupPath);
        return (!matches.isEmpty() ? new PatternsCondition(matches, this.pathHelper, this.pathMatcher) : null);
    }

    /**
     * Find the patterns matching the given lookup path. Invoking this method should
     * yield results equivalent to those of calling
     * {@link #getMatchingCondition(java.lang.String)}.
     * This method is provided as an alternative to be used if no request is available
     * (e.g. introspection, tooling, etc).
     *
     * @param lookupPath the lookup path to match to existing patterns
     * @return a collection of matching patterns sorted with the closest match at the top
     */
    public List<String> getMatchingPatterns(String lookupPath) {
        List<String> matches = new ArrayList<>();
        for (String pattern : this.patterns) {
            String match = getMatchingPattern(pattern, lookupPath);
            if (match != null) {
                matches.add(match);
            }
        }
        if (matches.size() > 1) {
            matches.sort(this.pathMatcher.getPatternComparator(lookupPath));
        }
        return matches;
    }

    @Nullable
    private String getMatchingPattern(String pattern, String lookupPath) {
        if (pattern.equals(lookupPath)) {
            return pattern;
        }
        boolean hasSuffix = pattern.indexOf('.') != -1;
        if (!hasSuffix && this.pathMatcher.match(pattern + ".*", lookupPath)) {
            return pattern + ".*";
        }
        if (this.pathMatcher.match(pattern, lookupPath)) {
            return pattern;
        }
        if (!pattern.endsWith("/") && this.pathMatcher.match(pattern + "/", lookupPath)) {
            return pattern + "/";
        }
        return null;
    }
}
