package com.digiwin.dap.middle.ram.support.iam;

import com.digiwin.dap.middle.ram.domain.remote.AppAuthResult;
import com.digiwin.dap.middle.ram.domain.remote.DevInfoVO;
import com.digiwin.dap.middle.ram.domain.sign.VerifyLog;
import com.digiwin.dap.middleware.cache.RedisUtils;

import java.time.Duration;
import java.time.LocalDateTime;

/**
 * 鉴权相关的业务缓存
 */
public class RedisCache {

    /**
     * 授权缓存:tenantId:userId:sysId
     */
    private static final String TENANT_USER_SYS_AUTH = "cac:tenant:user:auth:%s::%s::%s";

    /**
     * 平台的应用缓存 平台ID
     */
    public static final String SOP_PLATFORM_PREFIX = "entity:sysonplatform:platform:";
    /**
     * 应用的平台缓存 应用ID
     */
    public static final String SOP_SYS_PREFIX = "entity:sysonplatform:sys:";

    /**
     * 应用的服务商ID
     */
    public static final String ISV_DEV_TENANT_ID = "ram:api:subscribe:dev:%s";

    /**
     * 服务商访问凭证 devKey
     */
    public static final String ISV_CREDENTIAL = "entity:isv:credential:key:%s";

    /**
     * 服务商访问凭证最近使用信息 tenantSid:devKey
     */
    public static final String ISV_CREDENTIAL_RECORD = "ram:api:sign:verify:%s:%s";

    public static AppAuthResult getAuthResult(String tenantId, String userId, String sysId) {
        String cacAuthKey = String.format(TENANT_USER_SYS_AUTH, tenantId, userId, sysId);
        return RedisUtils.get(cacAuthKey, AppAuthResult.class);
    }

    public static void setAuthResult(AppAuthResult result, String userId, String sysId) {
        String cacAuthKey = String.format(TENANT_USER_SYS_AUTH, result.getTenantId(), userId, sysId);
        RedisUtils.set(cacAuthKey, result, calcDuration(result.getExpiredDateTime()));
    }

    /**
     * 缓存时间计算，最多缓存一周
     * 1、如果过期时间大于7天，则返回 7天
     * 2、如果过期时间小于7天，则返回 过期时间-当前时间
     *
     * @param expiredDateTime 过期时间
     */
    public static Duration calcDuration(LocalDateTime expiredDateTime) {
        LocalDateTime now = LocalDateTime.now();
        if (now.plusDays(7).isBefore(expiredDateTime)) {
            return Duration.ofDays(7);
        }
        return Duration.between(now, expiredDateTime);
    }

    public static DevInfoVO getDevTenant(String devId) {
        String key = String.format(ISV_DEV_TENANT_ID, devId);
        return RedisUtils.get(key, DevInfoVO.class);
    }

    public static void setDevTenant(String devId, DevInfoVO devInfoVO) {
        String key = String.format(ISV_DEV_TENANT_ID, devId);
        RedisUtils.set(key, devInfoVO, Duration.ofDays(1));
    }

    public static String getDevSecret(String devKey) {
        String key = String.format(ISV_CREDENTIAL, devKey);
        return RedisUtils.get(key, String.class);
    }

    public static void setDevSecret(String devKey, String appSecret) {
        String key = String.format(ISV_CREDENTIAL, devKey);
        RedisUtils.set(key, appSecret, Duration.ofDays(1));
    }

    public static void setIsvCredentialRecord(long tenantSid, String devKey, VerifyLog verifyLog) {
        String key = String.format(ISV_CREDENTIAL_RECORD, tenantSid, devKey);
        RedisUtils.set(key, verifyLog);
    }
}
