package com.digiwin.dap.middle.ram.service.access.apisub.handler;

import com.digiwin.dap.middle.kms.constants.KeyConstant;
import com.digiwin.dap.middle.ram.constant.RamI18nError;
import com.digiwin.dap.middle.ram.domain.PolicyOrder;
import com.digiwin.dap.middle.ram.domain.access.ApiSubAccessUser;
import com.digiwin.dap.middle.ram.domain.enums.ResultType;
import com.digiwin.dap.middle.ram.domain.remote.AuthApiVO;
import com.digiwin.dap.middle.ram.domain.remote.AuthCodeVO;
import com.digiwin.dap.middle.ram.domain.remote.AuthResultVO;
import com.digiwin.dap.middle.ram.domain.request.AccessResult;
import com.digiwin.dap.middle.ram.domain.request.AccessUser;
import com.digiwin.dap.middle.ram.service.access.apisub.RamApiSubPolicyHandler;
import com.digiwin.dap.middle.ram.support.RamHandlerSupport;
import com.digiwin.dap.middleware.commons.crypto.AES;
import com.digiwin.dap.middleware.constant.DapHttpHeaders;
import com.digiwin.dap.middleware.exception.BusinessException;
import com.digiwin.dap.middleware.exception.UnauthorizedException;
import org.springframework.core.Ordered;
import org.springframework.util.CollectionUtils;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;

public class RamAuthApiSubPolicyHandler extends RamApiSubPolicyHandler implements Ordered {

    private final RamHandlerSupport ramHandlerSupport;

    public RamAuthApiSubPolicyHandler(RamHandlerSupport ramHandlerSupport) {
        this.ramHandlerSupport = ramHandlerSupport;
    }

    @Override
    public AccessResult handle(ApiSubAccessUser request) {
        try {
            AuthCodeVO authCode = analysis(request);
            AuthResultVO authResult = ramHandlerSupport.getAuthResult(authCode.getTenantId(), authCode.getSysId(), request.getMethod(), request.getPath());
            if (isNotBelongToCurrentApi(request, authResult, authCode.getSysId())) {
                throw new BusinessException("应用[" + request.getSysId() + "]不属于API商品归属应用");
            }
            if (authResult.getPaymentType() == 0 && authResult.getRemainUsage() <= 0) {
                throw new BusinessException(RamI18nError.CAC_AUTH_COUNT_ZERO);
            }
            // api点数商品
            if ("APIQUOTA".equalsIgnoreCase(authResult.getCategory())) {
                if (authResult.getExpiredDateTime() == null || LocalDateTime.now().isAfter(authResult.getExpiredDateTime())) {
                    throw new BusinessException(RamI18nError.API_SUBSCRIBE_EXPIRED, new Object[]{authCode.getTenantId()});
                }
                if (CollectionUtils.isEmpty(authResult.getApis())) {
                    throw new BusinessException(RamI18nError.API_SUBSCRIBE_NONE, new Object[]{authCode.getTenantId(), request.getMethod(), request.getPath()});
                }
                AuthApiVO authApiVO = authResult.getApis().stream().filter(e -> e.getExpiredTime() != null && e.getExpiredTime().isAfter(LocalDateTime.now())).findFirst().orElse(null);
                if (authApiVO == null) {
                    throw new BusinessException(RamI18nError.API_SUBSCRIBE_EXPIRED, new Object[]{authCode.getTenantId()});
                }
                if (authResult.getPaymentType() == 4) {
                    AuthResultVO resourceGood = authResult.getResourceGood();
                    if (resourceGood == null || resourceGood.getRemainUsage() == null || resourceGood.getRemainUsage() <= 0) {
                        throw new BusinessException(RamI18nError.CAC_AUTH_COUNT_ZERO);
                    }
                }
            }
        } catch (Exception e) {
            throw new UnauthorizedException(ResultType.EXPLICIT_DENY, e.getMessage());
        }
        return AccessResult.next();
    }

    private static boolean isNotBelongToCurrentApi(AccessUser request, AuthResultVO authResult, String currentApiId) {
        String sysId = request.getSysId();

        // 如果sysId等于currentApiId，则认为是belong的，直接返回false
        if (sysId != null && sysId.equals(currentApiId)) {
            return false;
        }

        return (!"APIQUOTA".equalsIgnoreCase(authResult.getCategory())) && authResult.getApiGoods() != null
                && authResult.getApiGoods().stream().noneMatch(p -> p.getCode().equals(sysId));
    }

    private AuthCodeVO analysis(AccessUser request) {
        try {
            String secretKey = request.getHeaders().get(DapHttpHeaders.SECRET_KEY.getHeader());
            String[] content = AES.decryptCBC(secretKey, KeyConstant.OTHER).split("@#\\$%\\^_");
            if (content.length != 4) {
                throw new BusinessException("非法授权码：" + secretKey);
            }
            AuthCodeVO authCode = new AuthCodeVO();
            authCode.setTenantId(content[0]);
            authCode.setSysId(content[1]);
            authCode.setUserId(content[2]);
            Instant instant = Instant.ofEpochMilli(Long.parseLong(content[3]));
            authCode.setTimestamp(LocalDateTime.ofInstant(instant, ZoneId.systemDefault()));
            return authCode;
        } catch (Exception e) {
            throw new BusinessException(RamI18nError.IAM_AUTH_CODE_INVALID, new Object[]{e.getMessage()});
        }
    }

    @Override
    public int getOrder() {
        return PolicyOrder.API_SUB_AUTH.order();
    }
}
