package com.digiwin.dap.middle.ram.support.iam;

import com.digiwin.dap.middle.ram.constant.RamI18nError;
import com.digiwin.dap.middle.ram.domain.remote.AppAuthResult;
import com.digiwin.dap.middle.ram.domain.remote.AuthResultVO;
import com.digiwin.dap.middle.ram.domain.remote.DevInfoVO;
import com.digiwin.dap.middle.ram.domain.remote.Function;
import com.digiwin.dap.middle.ram.domain.request.AccessUser;
import com.digiwin.dap.middle.ram.support.RamCacheSupport;
import com.digiwin.dap.middle.ram.support.RamHandlerSupport;
import com.digiwin.dap.middleware.auth.domain.AuthResult;
import com.digiwin.dap.middleware.constant.BeanConstants;
import com.digiwin.dap.middleware.constant.GlobalConstants;
import com.digiwin.dap.middleware.constant.InternalUrl;
import com.digiwin.dap.middleware.domain.CommonErrorCode;
import com.digiwin.dap.middleware.domain.DapEnv;
import com.digiwin.dap.middleware.domain.StdData;
import com.digiwin.dap.middleware.exception.*;
import com.digiwin.dap.middleware.util.JsonUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 默认实现
 *
 * @author fobgochod
 */
public class DefaultRamHandlerSupport implements RamHandlerSupport {

    private static final Logger logger = LoggerFactory.getLogger(DefaultRamHandlerSupport.class);
    private static final String USER_WHITE_LIST = "integration;virtual";

    @Autowired
    private DapEnv dapEnv;
    @Resource(name = BeanConstants.DAP_RETRY_TEMPLATE)
    private RestTemplate retryRestTemplate;
    private RamCacheSupport ramCacheSupport;

    public DefaultRamHandlerSupport() {
    }

    public DefaultRamHandlerSupport(RamCacheSupport ramCacheSupport) {
        this.ramCacheSupport = ramCacheSupport;
    }

    @Override
    public AuthResult getAuthResult(AuthResult authResult, AccessUser request) {
        AppAuthResult result = RedisCache.getAuthResult(request.getTenantId(), request.getUserId(), request.getSysId());
        if (result == null) {
            result = getAuthResult(request.getTenantId(), request.getUserId(), request.getSysId(),
                    request.isPlatform(), authResult.getTenants(),
                    !USER_WHITE_LIST.contains(request.getUserId()));

            if (result.getSuccess()) {
                RedisCache.setAuthResult(result, request.getUserId(), request.getSysId());
            }
        }
        return result.flushAuthType();
    }

    @Override
    public List<String> getUserFunction(AccessUser request) {
        return getUserFunction(request.getUserId(), request.getTenantId(), request.getSysId(), request.getUserToken())
                .uris(request.getSysId());
    }

    @Override
    public AuthResultVO getAuthResult(String tenantId, String appId, String method, String path) {
        String url = dapEnv.getCacUri() + InternalUrl.TENANT_GOODS_CHECK;
        Map<String, Object> body = new HashMap<>();
        try {
            body.put("tenantId", tenantId);
            body.put("goodsCode", appId);
            body.put("method", method);
            body.put("path", path);
            HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(body);
            ResponseEntity<StdData<AuthResultVO>> result = retryRestTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<StdData<AuthResultVO>>() {
            });
            return result.getBody().getData();
        } catch (HttpStatusCodeException e) {
            StdError stdError = JsonUtils.jsonToObj(e.getResponseBodyAsString(), StdError.class);
            throw new UnauthorizedException(stdError, url, e);
        } catch (Exception e) {
            throw new ThirdCallException(CommonErrorCode.REMOTE_UNEXPECTED, url, e);
        }
    }


    @Override
    public String getDevSecret(String userToken, String devKey) {
        String result = RedisCache.getDevSecret(devKey);
        if (result != null) {
            return result;
        }
        try {
            String url = dapEnv.getIamUri() + InternalUrl.TENANT_SECRET_CURRENT;
            Map<String, Object> body = new HashMap<>();
            body.put("devKey", devKey);
            HttpHeaders headers = new HttpHeaders();
            headers.add(GlobalConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);
            HttpEntity<Object> request = new HttpEntity<>(body, headers);

            ResponseEntity<StdData<DevInfoVO>> responseEntity = retryRestTemplate.exchange(url, HttpMethod.POST, request, new ParameterizedTypeReference<StdData<DevInfoVO>>() {
            });
            String appSecret = responseEntity.getBody().getData().getAppSecret();
            if (appSecret == null) {
                throw new BusinessException(RamI18nError.API_DEV_SECRET_NONE, new Object[]{devKey});
            }
            RedisCache.setDevSecret(devKey, appSecret);
            return appSecret;
        } catch (DapException e) {
            throw e;
        } catch (HttpStatusCodeException e) {
            StdError stdError = JsonUtils.jsonToObj(e.getResponseBodyAsString(), StdError.class);
            throw new BusinessException(stdError);
        } catch (Exception e) {
            throw new BusinessException("【调用IAM】获取devSecret失败", e);
        }
    }

    @Override
    public DevInfoVO getDevTenant(String userToken, String devId) {
        DevInfoVO result = RedisCache.getDevTenant(devId);
        if (result != null) {
            return result;
        }
        try {
            String url = dapEnv.getIamUri() + InternalUrl.DEV_APP_TENANT;
            Map<String, Object> body = new HashMap<>();
            body.put("id", devId);
            HttpHeaders headers = new HttpHeaders();
            headers.add(GlobalConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);
            HttpEntity<Object> request = new HttpEntity<>(body, headers);

            ResponseEntity<StdData<DevInfoVO>> responseEntity = retryRestTemplate.exchange(url, HttpMethod.POST, request, new ParameterizedTypeReference<StdData<DevInfoVO>>() {
            });
            DevInfoVO devInfoVO = responseEntity.getBody().getData();
            if (devInfoVO == null) {
                throw new BusinessException(RamI18nError.API_DEV_TENANT_NONE, new Object[]{devId});
            }
            RedisCache.setDevTenant(devId, devInfoVO);
            return devInfoVO;
        } catch (DapException e) {
            throw e;
        } catch (HttpStatusCodeException e) {
            StdError stdError = JsonUtils.jsonToObj(e.getResponseBodyAsString(), StdError.class);
            throw new BusinessException(stdError);
        } catch (Exception e) {
            throw new BusinessException(String.format("【调用IAM】获取应用[%s]的开发商信息失败", devId), e);
        }
    }


    private AppAuthResult getAuthResult(String tenantId, String userId,
                                        String appId, boolean platform,
                                        List<String> tenants, boolean checkUserAuth) {
        try {
            String url = dapEnv.getCacUri() + InternalUrl.AUTH_CHECK;
            Map<String, Object> body = new HashMap<>();
            body.put("tenantId", tenantId);
            body.put("userId", userId);
            body.put("goodsCode", appId);
            body.put("tenants", tenants);
            body.put("checkUserAuth", Boolean.toString(checkUserAuth));
            if (platform) {
                // 是平台，则查询旗下的应用
                body.put("apps", ramCacheSupport.getSysByPlatformId(appId));
            } else {
                // 是应用，则查询该应用的平台
                body.put("platformApps", ramCacheSupport.getPlatformSysBySysId(appId));
            }
            HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(body);
            return retryRestTemplate.postForObject(url, httpEntity, AppAuthResult.class);
        } catch (Exception e) {
            String message = String.format("获取租户[%s]用户[%s]的应用[%s]授权信息失败", tenantId, userId, appId);
            throw new UnauthorizedException(RamI18nError.CAC_AUTH_GET_FAIL, message, 411005, e);
        }
    }

    private Function getUserFunction(String userId, String tenantId, String sysId, String userToken) {
        String uri = dapEnv.getIamUri() + InternalUrl.USER_PERMISSION;
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            headers.add(GlobalConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);

            Map<String, String> body = new HashMap<>(3);
            body.put("tenantId", tenantId);
            body.put("userId", userId);
            body.put("target", "drn:iam:app:" + sysId);
            HttpEntity<Map<String, String>> request = new HttpEntity<>(body, headers);
            return retryRestTemplate.postForObject(uri, request, Function.class);
        } catch (Exception e) {
            throw new ThirdCallException(CommonErrorCode.REMOTE_UNEXPECTED, uri, e);
        }
    }
}
