package com.digiwin.dap.middle.ram.service.access.api.handler;

import com.digiwin.dap.middle.ram.domain.PolicyOrder;
import com.digiwin.dap.middle.ram.domain.access.ApiAccessUser;
import com.digiwin.dap.middle.ram.domain.console.policy.PolicyGrantVO;
import com.digiwin.dap.middle.ram.domain.enums.ApiPolicyType;
import com.digiwin.dap.middle.ram.domain.enums.ResultType;
import com.digiwin.dap.middle.ram.domain.enums.TargetType;
import com.digiwin.dap.middle.ram.domain.request.AccessResult;
import com.digiwin.dap.middle.ram.domain.sign.RamSignInfo;
import com.digiwin.dap.middle.ram.domain.sign.VerifyLog;
import com.digiwin.dap.middle.ram.domain.vo.PatternVO;
import com.digiwin.dap.middle.ram.service.access.api.RamApiPolicyHandler;
import com.digiwin.dap.middle.ram.service.core.RamBaseGroupService;
import com.digiwin.dap.middle.ram.support.RamHandlerSupport;
import com.digiwin.dap.middle.ram.support.iam.RedisCache;
import com.digiwin.dap.middle.ram.support.web.MappingRegistry;
import com.digiwin.dap.middle.ram.util.MatcherUtils;
import com.digiwin.dap.middleware.commons.crypto.SignUtils;
import com.digiwin.dap.middleware.constant.DapHttpHeaders;
import com.digiwin.dap.middleware.domain.CommonErrorCode;
import com.digiwin.dap.middleware.exception.UnauthorizedException;
import com.digiwin.dap.middleware.util.JsonUtils;
import com.digiwin.dap.middleware.util.VerifyUtils;
import org.springframework.core.Ordered;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;

/**
 * API访问控制，策略类型 {@link ApiPolicyType#ApiAuthSign}
 * <p>
 * 该策略下的API需要检测授权
 *
 * @author fobgochod
 * @since 4.36.3
 */
@Service
public class RamAuthSignApiPolicyHandler extends RamApiPolicyHandler implements Ordered {

    private final RamHandlerSupport ramHandlerSupport;

    public RamAuthSignApiPolicyHandler(RamBaseGroupService baseGroupService, RamHandlerSupport ramHandlerSupport) {
        super(baseGroupService);
        this.ramHandlerSupport = ramHandlerSupport;
    }

    @Override
    public AccessResult handle(ApiAccessUser request) {
        MappingRegistry mappingRegistry = ramBaseGroupService.getPattern(request.getApp(), request.getAppId(), ApiPolicyType.ApiAuthSign.name());
        PatternVO matched = MatcherUtils.matches(request.getMethod(), request.getPath(), mappingRegistry);
        if (matched == null) {
            return AccessResult.allow();
        }
        try {
            List<PolicyGrantVO> grants = ramBaseGroupService.findGrantsByRoute(request.getApp(), TargetType.Tenant.name(), matched.getMethod(), matched.getPath());
            PolicyGrantVO.checkValidGrant(grants, request);
            verifySign(request);
        } catch (Exception e) {
            throw new UnauthorizedException(ResultType.EXPLICIT_DENY, e.getMessage());
        }
        return AccessResult.allow();
    }

    private void verifySign(ApiAccessUser request) {
        String signHeader = request.getHeaders().get(DapHttpHeaders.DEV_ARGS.getHeader());
        RamSignInfo signInfo = RamSignInfo.get(signHeader, RamSignInfo.class);
        VerifyUtils.sign(signInfo, () -> true);
        String devKey = request.getHeaders().get(DapHttpHeaders.DEV_ID.getHeader());
        String devSecret = ramHandlerSupport.getDevSecret(request.getUserToken(), devKey);
        boolean verified = SignUtils.verify(JsonUtils.objToMap(signInfo), devSecret);
        if (verified) {
            verifyLog(devKey, devSecret, request);
        } else {
            throw new UnauthorizedException(CommonErrorCode.SIGN_INCONSISTENT_SIGNATURE);
        }
    }

    /**
     * 记录开发商凭证最近使用信息
     *
     * @param devKey    秘钥KEY
     * @param devSecret 密钥
     * @param request   请求信息
     */
    private void verifyLog(String devKey, String devSecret, ApiAccessUser request) {
        VerifyLog verifyLog = new VerifyLog();
        verifyLog.setVerifyDate(LocalDateTime.now());
        verifyLog.setTenantSid(request.getTenantSid());
        verifyLog.setUserSid(request.getUserSid());
        verifyLog.setSysSid(request.getSysSid());
        verifyLog.setSysId(request.getSysId());
        verifyLog.setDevKey(devKey);
        verifyLog.setDevSecret(devSecret);
        RedisCache.setIsvCredentialRecord(request.getTenantSid(), devKey, verifyLog);
    }

    @Override
    public int getOrder() {
        return PolicyOrder.API_AUTH_SIGN.order();
    }
}