package com.digiwin.dap.middle.ram.support.security.path.handler;

import com.digiwin.dap.middle.ram.constant.RamI18nError;
import com.digiwin.dap.middle.ram.domain.enums.PolicyType;
import com.digiwin.dap.middle.ram.domain.request.AccessResult;
import com.digiwin.dap.middle.ram.domain.request.AccessUser;
import com.digiwin.dap.middle.ram.domain.vo.PatternVO;
import com.digiwin.dap.middle.ram.service.core.RamCoreService;
import com.digiwin.dap.middle.ram.support.RamHandlerSupport;
import com.digiwin.dap.middle.ram.support.security.path.PathPolicyHandler;
import com.digiwin.dap.middle.ram.support.web.MappingRegistry;
import com.digiwin.dap.middle.ram.util.MatcherUtils;
import com.digiwin.dap.middleware.commons.util.StrUtils;
import com.digiwin.dap.middleware.constant.GlobalConstants;
import com.digiwin.dap.middleware.exception.UnauthorizedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;

import java.util.List;

/**
 * 功能权限，策略类型 {@link PolicyType#Function}
 * <p>
 * 用户拥有对应功能权限即拥有对应的API访问权限
 * <p>
 * 此类型下策略ID即作业(actionId)或状态ID(conditionId)
 *
 * @author fobgochod
 */
@Order(400)
public class FunctionPathPolicyHandler extends PathPolicyHandler<AccessUser> {

    private static final Logger logger = LoggerFactory.getLogger(FunctionPathPolicyHandler.class);

    private final RamHandlerSupport ramHandlerSupport;

    public FunctionPathPolicyHandler(RamCoreService ramCoreService, RamHandlerSupport ramHandlerSupport) {
        super(ramCoreService);
        this.ramHandlerSupport = ramHandlerSupport;
    }

    @Override
    public AccessResult handle(AccessUser request) {
        if (request.getUserId() != null && request.getTenantId() != null && request.getSysId() != null) {
            // 查询当前访问应用sysId有没有配置功能权限
            MappingRegistry mappingRegistry = ramCoreService.getPattern(request.getSysId(), PolicyType.Function.name());
            PatternVO pattern = MatcherUtils.matches(request.getMethod(), request.getPath(), mappingRegistry);
            if (pattern != null) {
                List<String> policyIds = ramCoreService.findPolicyIdByRoute(request.getSysId(), PolicyType.Function.name(), pattern.getMethod(), pattern.getPath());
                if (!policyIds.isEmpty()) {
                    boolean match = ramHandlerSupport.getUserFunction(request).stream().anyMatch(policyIds::contains);
                    if (match) {
                        return AccessResult.allow();
                    }
                }
                handleException(request);
            }
        }
        return AccessResult.next();
    }

    private void handleException(AccessUser request) {
        logger.error("\n{}", request.securityLog());
        String errMsg = String.format("用户[%s]没有权限访问应用[%s]接口[%s-%s]%s",
                request.getUserId(), request.getAppId(), request.getMethod(), request.getPath(),
                StrUtils.isEmpty(request.getTableName()) ? GlobalConstants.EMPTY : String.format("(%s)", request.getTableName()));
        throw new UnauthorizedException(RamI18nError.CYBER_SECURITY, errMsg);
    }
}
