package com.digiwin.dap.middle.ram.filter;

import com.digiwin.dap.middle.ram.domain.request.AccessResult;
import com.digiwin.dap.middle.ram.domain.request.AccessUser;
import com.digiwin.dap.middle.ram.support.security.auth.AuthPolicyHandler;
import com.digiwin.dap.middle.ram.support.security.path.PathPolicyHandler;
import com.digiwin.dap.middleware.auth.AppAuthContext;
import com.digiwin.dap.middleware.auth.AppAuthContextHolder;
import com.digiwin.dap.middleware.auth.domain.RamVersion;
import com.digiwin.dap.middleware.auth.domain.RequestInfo;
import com.digiwin.dap.middleware.constant.InternalUrl;
import com.digiwin.dap.middleware.domain.FilterOrderEnum;
import org.springframework.core.Ordered;
import org.springframework.web.filter.OncePerRequestFilter;

import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * 访问控制RAM（Resource Access Management）<br>
 * 管理用户身份与资源访问权限
 *
 * @author fobgochod
 * @see RamVersion#v1
 */
public class RamV1AccessCheckFilter extends OncePerRequestFilter implements Ordered {

    private final PathPolicyHandler<AccessUser> pathPolicyHandler;
    private final AuthPolicyHandler authPolicyHandler;

    public RamV1AccessCheckFilter(PathPolicyHandler<AccessUser> pathPolicyHandler, AuthPolicyHandler authPolicyHandler) {
        this.pathPolicyHandler = pathPolicyHandler;
        this.authPolicyHandler = authPolicyHandler;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        String requestURI = request.getRequestURI();
        final AppAuthContext context = AppAuthContextHolder.getContext();
        RequestInfo requestInfo = context.getRequestInfo();
        if (InternalUrl.token().contains(requestURI)) {
            RamVersion version = RamVersion.of(request.getParameter("ramVersion"));
            if (version == RamVersion.v1) {
                ramCheckV1(context, requestInfo);
            }
        } else if (context.isInternal() || InternalUrl.auth().contains(requestURI) || InternalUrl.login().contains(requestURI)) {
            // 控制流程的接口，不能继续走控制流程，防止陷入无线循环，参考说明 RamV2AccessCheckFilter
        } else {
            ramCheckV1(context, requestInfo);
        }
        filterChain.doFilter(request, response);
    }

    /**
     * 等价于RAM服务接口 {@link InternalUrl#BASE_ANALYZE}
     */
    private void ramCheckV1(final AppAuthContext context, RequestInfo requestInfo) {
        AccessUser requestUser = AccessUser.of(requestInfo.getAppId(), requestInfo.getMethod(), requestInfo.getPath());
        // 接口访问控制
        AccessResult accessResult = pathPolicyHandler.handle(requestUser);
        accessResult.checkAccessResult();
        context.setResultType(accessResult.getResultType().name());
        // 授权访问控制
        authPolicyHandler.handle(requestUser);
    }

    @Override
    public int getOrder() {
        return FilterOrderEnum.ACCESS_CHECK.order();
    }
}
