package com.digiwin.dap.middle.ram.service.access.api.handler;

import com.digiwin.dap.middle.ram.constant.RamI18nError;
import com.digiwin.dap.middle.ram.domain.PolicyOrder;
import com.digiwin.dap.middle.ram.domain.Statement;
import com.digiwin.dap.middle.ram.domain.access.ApiAccessUser;
import com.digiwin.dap.middle.ram.domain.enums.ApiPolicyType;
import com.digiwin.dap.middle.ram.domain.enums.TargetType;
import com.digiwin.dap.middle.ram.domain.remote.DevInfoVO;
import com.digiwin.dap.middle.ram.domain.request.AccessResult;
import com.digiwin.dap.middle.ram.domain.vo.PatternVO;
import com.digiwin.dap.middle.ram.service.access.api.RamApiPolicyHandler;
import com.digiwin.dap.middle.ram.service.core.RamBaseGroupService;
import com.digiwin.dap.middle.ram.support.RamHandlerSupport;
import com.digiwin.dap.middle.ram.util.MatcherUtils;
import com.digiwin.dap.middleware.exception.BusinessException;
import org.springframework.core.Ordered;
import org.springframework.stereotype.Service;

/**
 * API访问控制，策略类型 {@link ApiPolicyType#ApiAccessDeny}
 * <p>
 * 该策略下的API需要检测授权
 *
 * @author fobgochod
 * @since 4.36.3
 */
@Service
public class RamAccessApiPolicyHandler extends RamApiPolicyHandler implements Ordered {

    private final RamHandlerSupport ramHandlerSupport;

    public RamAccessApiPolicyHandler(RamBaseGroupService baseGroupService, RamHandlerSupport ramHandlerSupport) {
        super(baseGroupService);
        this.ramHandlerSupport = ramHandlerSupport;
    }

    @Override
    public AccessResult handle(ApiAccessUser request) {
        if (request.getUserToken() == null || request.getAppToken() == null) {
            // TODO 像DMC预览、各种中间件各种白名单API，没有token信息，先跳过。
            return AccessResult.allow();
        }
        // 根据应用白名单来区分是否需要API订阅权限管控
        boolean isGrant = ramBaseGroupService.existGrant(request.getApp(), TargetType.Sys.name(), ApiPolicyType.API_ACCESS_ALLOW_ID, request.getSysId());
        if (isGrant) {
            return AccessResult.allow();
        }
        Statement denys = ramBaseGroupService.getPolicyRoute(request.getApp(), request.getAppId(), ApiPolicyType.ApiAccessDeny.name(), TargetType.Sys.name(), request.getSysId());
        PatternVO denyMatched = MatcherUtils.matches(request.getMethod(), request.getPath(), denys.getAllow());
        if (denyMatched != null) {
            throw new BusinessException(RamI18nError.API_POLICY_ACCESS_DENY, new Object[]{request.getSysId()});
            // return ResultType.EXPLICIT_DENY;
        }
        // 接下来要进行各种认证，userToken必定存在
        DevInfoVO appDev = ramHandlerSupport.getDevTenant(request.getUserToken(), request.getSysId());
        request.setDevTenantId(appDev.getTenantId());
        return AccessResult.next();
    }

    @Override
    public int getOrder() {
        return PolicyOrder.API_ACCESS.order();
    }
}
