package com.digiwin.dap.middle.ram.support.web.condition;

import com.digiwin.dap.middleware.commons.util.StrUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.PathMatcher;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.util.UrlPathHelper;

import jakarta.servlet.http.HttpServletRequest;
import java.util.Set;

/**
 * Request mapping information. Encapsulates the following request mapping conditions:
 * <ol>
 * <li>{@link PatternsCondition}
 * <li>{@link MethodsCondition}
 * </ol>
 *
 * @author fobgochod
 */
public final class MappingInfo {

    @Nullable
    private final String name;

    private final PatternsCondition patternsCondition;

    private final MethodsCondition methodsCondition;


    public MappingInfo(@Nullable String name, @Nullable PatternsCondition patterns, @Nullable MethodsCondition methods) {

        this.name = (StrUtils.hasText(name) ? name : null);
        this.patternsCondition = (patterns != null ? patterns : new PatternsCondition());
        this.methodsCondition = (methods != null ? methods : new MethodsCondition());
    }

    /**
     * Creates a new instance with the given request conditions.
     */
    public MappingInfo(@Nullable PatternsCondition patterns, @Nullable MethodsCondition methods) {

        this(null, patterns, methods);
    }

    /**
     * Re-create a RequestMappingInfo with the given custom request condition.
     */
    public MappingInfo(MappingInfo info) {
        this(info.name, info.patternsCondition, info.methodsCondition);
    }

    /**
     * Create a new {@code RequestMappingInfo.Builder} with the given paths.
     *
     * @param paths the paths to use
     * @since 4.2
     */
    public static Builder paths(String... paths) {
        return new DefaultBuilder(paths);
    }

    /**
     * Return the HTTP request methods of this {@link MappingInfo};
     * or instance with 0 request methods (return {@code []}).
     */
    public static HttpMethod[] getMethod(String method) {
        try {
            return new HttpMethod[]{HttpMethod.valueOf(method)};
        } catch (Exception e) {
            return new HttpMethod[0];
        }
    }

    /**
     * Return the name for this mapping, or {@code null}.
     */
    @Nullable
    public String getName() {
        return this.name;
    }

    /**
     * Return the URL patterns of this {@link MappingInfo};
     * or instance with 0 patterns (never {@code null}).
     */
    public PatternsCondition getPatternsCondition() {
        return this.patternsCondition;
    }

    /**
     * Return the HTTP request methods of this {@link MappingInfo};
     * or instance with 0 request methods (never {@code null}).
     */
    public MethodsCondition getMethodsCondition() {
        return this.methodsCondition;
    }


    /**
     * Checks if all conditions in this request mapping info match the provided request and returns
     * a potentially new request mapping info with conditions tailored to the current request.
     * <p>For example the returned instance may contain the subset of URL patterns that match to
     * the current request, sorted with best matching patterns on top.
     *
     * @return a new instance in case all conditions match; or {@code null} otherwise
     */
    public MappingInfo getMatchingCondition(String method, String lookupPath) {
        MethodsCondition methods = this.methodsCondition.getMatchingCondition(method);
        if (methods == null) {
            return null;
        }

        PatternsCondition patterns = this.patternsCondition.getMatchingCondition(lookupPath);
        if (patterns == null) {
            return null;
        }

        return new MappingInfo(this.name, patterns, methods);
    }

    @Nullable
    public MappingInfo getMatchingCondition(HttpServletRequest request) {
        MethodsCondition methods = this.methodsCondition.getMatchingCondition(request);
        if (methods == null) {
            return null;
        }

        PatternsCondition patterns = this.patternsCondition.getMatchingCondition(request);
        if (patterns == null) {
            return null;
        }

        return new MappingInfo(this.name, patterns, methods);
    }

    @Override
    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (!(other instanceof MappingInfo)) {
            return false;
        }
        MappingInfo otherInfo = (MappingInfo) other;
        return (this.patternsCondition.equals(otherInfo.patternsCondition) && this.methodsCondition.equals(otherInfo.methodsCondition));
    }

    @Override
    public int hashCode() {
        return (this.patternsCondition.hashCode() * 31 +  // primary differentiation
                this.methodsCondition.hashCode());
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder("{");
        if (!this.methodsCondition.isEmpty()) {
            Set<HttpMethod> httpMethods = this.methodsCondition.getMethods();
            builder.append(httpMethods.size() == 1 ? httpMethods.iterator().next() : httpMethods);
        }
        if (!this.patternsCondition.isEmpty()) {
            Set<String> patterns = this.patternsCondition.getPatterns();
            builder.append(" ").append(patterns.size() == 1 ? patterns.iterator().next() : patterns);
        }
        builder.append('}');
        return builder.toString();
    }


    /**
     * Defines a builder for creating a RequestMappingInfo.
     *
     * @since 4.2
     */
    public interface Builder {

        /**
         * Set the path patterns.
         */
        Builder paths(String... paths);

        /**
         * Set the request method conditions.
         */
        Builder methods(HttpMethod... methods);

        /**
         * Provide additional configuration needed for request mapping purposes.
         */
        Builder options(BuilderConfiguration options);

        /**
         * Build the RequestMappingInfo.
         */
        MappingInfo build();
    }


    private static class DefaultBuilder implements Builder {

        private String[] paths = new String[0];

        private HttpMethod[] methods = new HttpMethod[0];

        @Nullable
        private String mappingName;

        private BuilderConfiguration options = new BuilderConfiguration();

        public DefaultBuilder(String... paths) {
            this.paths = paths;
        }

        @Override
        public Builder paths(String... paths) {
            this.paths = paths;
            return this;
        }

        @Override
        public DefaultBuilder methods(HttpMethod... methods) {
            this.methods = methods;
            return this;
        }

        @Override
        public Builder options(BuilderConfiguration options) {
            this.options = options;
            return this;
        }

        @Override
        public MappingInfo build() {
            PatternsCondition patternsCondition = new PatternsCondition(this.paths, this.options.getUrlPathHelper(), this.options.getPathMatcher());

            return new MappingInfo(this.mappingName, patternsCondition, new MethodsCondition(this.methods));
        }
    }

    /**
     * Container for configuration options used for request mapping purposes.
     * Such configuration is required to create RequestMappingInfo instances but
     * is typically used across all RequestMappingInfo instances.
     *
     * @see RequestMappingInfo.Builder#options
     */
    public static class BuilderConfiguration {

        public static final UrlPathHelper defaultInstance = new UrlPathHelper();
        private UrlPathHelper urlPathHelper;
        private PathMatcher pathMatcher;

        public UrlPathHelper getUrlPathHelper() {
            return this.urlPathHelper;
        }

        public void setUrlPathHelper(UrlPathHelper urlPathHelper) {
            this.urlPathHelper = urlPathHelper;
        }

        public PathMatcher getPathMatcher() {
            return this.pathMatcher;
        }

        public void setPathMatcher(PathMatcher pathMatcher) {
            this.pathMatcher = pathMatcher;
        }
    }
}
