package com.digiwin.dap.middle.tx.stream.producer;

import com.digiwin.dap.middle.stream.domain.MessageBody;
import com.digiwin.dap.middle.stream.domain.MessageType;
import com.digiwin.dap.middle.stream.producer.channel.InternalOutput;
import com.digiwin.dap.middle.tx.stream.domain.TxMessageStatus;
import com.digiwin.dap.middle.tx.stream.entity.TransactionMessage;
import com.digiwin.dap.middle.tx.stream.service.TransactionMessageService;
import com.digiwin.dap.middleware.constant.DapHttpHeaders;
import com.digiwin.dap.middleware.constant.GlobalConstants;
import com.digiwin.dap.middleware.domain.DapEnv;
import com.digiwin.dap.middleware.util.JsonUtils;
import com.digiwin.dap.middleware.util.UserUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.messaging.Message;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.Optional;

@Service
public class TxProducerServiceImpl implements TxProducerService {

    @Autowired
    private DapEnv dapEnv;
    @Autowired(required = false)
    private InternalOutput internalChannel;
    @Autowired
    private TransactionMessageService transactionMessageService;

    @Override
    public void internal(MessageBody messageInfo) {
        internal(messageInfo, false);
    }

    @Override
    public void internal(MessageBody messageInfo, boolean transaction) {
        internal(messageInfo, transaction, LocalDateTime.now());
    }

    @Override
    public void internal(MessageBody messageBody, boolean transaction, LocalDateTime sendTime) {
        messageBody.setSourceId(dapEnv.getAppName());
        messageBody.setMsgType(MessageType.INTERNAL);
        if (messageBody.getUserToken() == null) {
            messageBody.setUserToken(UserUtils.getToken());
        }
        String payload = JsonUtils.writeValue(messageBody);
        if (payload == null) {
            return;
        }
        MessageBuilder<String> builder = MessageBuilder.withPayload(payload)
                .setHeader(DapHttpHeaders.MESSAGE_EVENT_ID.getHeader(), messageBody.getOpType())
                .setHeader(DapHttpHeaders.USER_TOKEN.getHeader(), messageBody.getUserToken())
                .setHeader(DapHttpHeaders.APP_TOKEN.getHeader(), GlobalConstants.DigiwinCloud_APP_TOKEN)
                .setHeader(DapHttpHeaders.ACCEPT_LANGUAGE.getHeader(), LocaleContextHolder.getLocale().toLanguageTag())
                .setHeader("userId", UserUtils.getUserId())
                .setHeader("tenantId", UserUtils.getTenantId());
        Optional.ofNullable(messageBody.getHeaderTag())
                .filter(tag -> !tag.isEmpty())
                .ifPresent(tag -> builder.setHeader("tag", tag));
        Message<?> message = builder.build();
        if (transaction) {
            TransactionMessage txMessage = new TransactionMessage();
            txMessage.setContent(JsonUtils.objToJson(messageBody));
            txMessage.setStatus(TxMessageStatus.PENDING.name());
            txMessage.setSendTime(sendTime);
            txMessage.setRetryCount(0);
            transactionMessageService.insertTransactionMessage(txMessage);
        } else {
            internalChannel.internal().send(message);
        }
    }
}
