package com.digiwin.dap.middle.tx.stream.schedule;

import com.digiwin.dap.middle.stream.domain.MessageBody;
import com.digiwin.dap.middle.tx.stream.entity.TransactionMessage;
import com.digiwin.dap.middle.tx.stream.producer.TxProducerService;
import com.digiwin.dap.middle.tx.stream.service.TransactionMessageService;
import com.digiwin.dap.middleware.util.JsonUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.List;

@ConditionalOnProperty(name = "dap.middleware.stream.tx-task", havingValue = "true", matchIfMissing = true)
@Service
public class TransactionMessageSchedule {

    private static final Logger logger = LoggerFactory.getLogger(TransactionMessageSchedule.class);

    private static final int PAGE_SIZE = 100;

    private static final int MAX_RETRY_COUNT = 3;

    @Autowired
    private TransactionMessageService transactionMessageService;

    @Autowired
    private TxProducerService txProducerService;

    @Scheduled(fixedDelay = 1000)
    public void scheduleMessage() {
        try {
            while (true) {
                List<TransactionMessage> txMessages = transactionMessageService.selectPendingTransactionMessage(PAGE_SIZE, MAX_RETRY_COUNT);
                if (CollectionUtils.isEmpty(txMessages)) {
                    return;
                }
                for (TransactionMessage txMessage : txMessages) {
                    int update = transactionMessageService.updateTransactionMessageStatusToSending(txMessage.getSid());
                    if (update < 1) {
                        continue;
                    }
                    try {
                        MessageBody messageBody = JsonUtils.jsonToObj(txMessage.getContent(), MessageBody.class);
                        txProducerService.internal(messageBody);
                    } catch (Exception e) {
                        logger.error("scheduleTransactionMessage error", e);
                        transactionMessageService.updateTransactionMessageStatusToPending(txMessage.getSid(), LocalDateTime.now().plusSeconds(5));
                        continue;
                    }
                    transactionMessageService.deleteTransactionMessage(txMessage.getSid());
                }
            }
        } catch (Exception e) {
            logger.error("scheduleTransactionMessage error", e);
        }
    }
}
