package com.digiwin.dap.middleware.dmc;

import com.digiwin.dap.middleware.dmc.common.comm.DefaultServiceClient;
import com.digiwin.dap.middleware.dmc.common.context.TenantHolder;
import com.digiwin.dap.middleware.dmc.common.utils.DMCUtils;
import com.digiwin.dap.middleware.dmc.common.utils.FileCopyUtils;
import com.digiwin.dap.middleware.dmc.common.utils.FileUtils;
import com.digiwin.dap.middleware.dmc.internal.DMCConfig;
import com.digiwin.dap.middleware.dmc.internal.DMCConstants;
import com.digiwin.dap.middleware.dmc.internal.DMCHeaders;
import com.digiwin.dap.middleware.dmc.internal.client.OSSClient;
import com.digiwin.dap.middleware.dmc.internal.operation.*;
import com.digiwin.dap.middleware.dmc.model.*;
import com.digiwin.dap.middleware.util.JsonUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * The entry point class of DMC that implements the DMC interface.
 *
 * @author fobgochod
 * @since 3.1.0.0
 */
public class DMCClient extends OSSClient implements DMC {

    private static final Logger logger = LoggerFactory.getLogger(DMCClient.class);

    private DMCLoginOperation loginOperation;
    private DMCUploadOperation uploadOperation;
    private DMCDownloadOperation downloadOperation;
    private DMCDeleteOperation deleteOperation;
    private DMCRecycleOperation recycleOperation;
    private DMCCopyOperation copyOperation;
    private DMCMoveOperation moveOperation;
    private DMCShareOperation shareOperation;
    private DMCImageOperation imageOperation;
    private DMCBaseOperation baseOperation;

    DMCClient(DMCConfig config) {
        this(config, null);
    }

    DMCClient(DMCConfig config, ClientConfiguration clientConfig) {
        DMCConfig.check(config);
        if (clientConfig == null) {
            this.serviceClient = new DefaultServiceClient(config);
        } else {
            this.serviceClient = new DefaultServiceClient(config, clientConfig);
        }
        initOperations();

        this.serviceClient.getAuthCache().setLoginOperation(loginOperation);
        if (!config.isLazy()) {
            this.serviceClient.setAuthCache(config.getTenantId());
        }
    }

    protected static List<FileInfo> toFileInfo(List<String> filenames) {
        List<FileInfo> fileInfos = new ArrayList<>();
        filenames.forEach(filename -> fileInfos.add(new FileInfo(filename)));
        return fileInfos;
    }

    protected static List<FileInfo> toFileInfo(Map<String, FileInfo> filenames) {
        List<FileInfo> fileInfos = new ArrayList<>();
        for (Map.Entry<String, FileInfo> entry : filenames.entrySet()) {
            FileInfo fileInfo = entry.getValue();
            fileInfo.setFileName(entry.getKey());
            fileInfos.add(fileInfo);
        }
        return fileInfos;
    }

    public void initOperations() {
        super.initOperations();
        this.loginOperation = new DMCLoginOperation(this.serviceClient);
        this.uploadOperation = new DMCUploadOperation(this.serviceClient);
        this.downloadOperation = new DMCDownloadOperation(this.serviceClient);
        this.deleteOperation = new DMCDeleteOperation(this.serviceClient);
        this.recycleOperation = new DMCRecycleOperation(this.serviceClient);
        this.copyOperation = new DMCCopyOperation(this.serviceClient);
        this.moveOperation = new DMCMoveOperation(this.serviceClient);
        this.shareOperation = new DMCShareOperation(this.serviceClient);
        this.imageOperation = new DMCImageOperation(this.serviceClient);
        this.baseOperation = new DMCBaseOperation(this.serviceClient);
    }

    @Override
    public String copy(String fileId) throws Exception {
        return this.copy(fileId, DMCConstants.EMPTY_UUID_STR);
    }

    @Override
    public String copy(String fileId, String targetDirId) throws Exception {
        FileOpRequest request = new FileOpRequest();
        request.setFileId(fileId);
        request.setTargetDirId(targetDirId);
        return copyOperation.copy(request);
    }

    @Override
    public String copyDir(String dirId) throws Exception {
        return this.copyDir(dirId, DMCConstants.EMPTY_UUID_STR);
    }

    @Override
    public String copyDir(String dirId, String targetDirId) throws Exception {
        FileOpRequest request = new FileOpRequest();
        request.setDirId(dirId);
        request.setTargetDirId(targetDirId);
        return copyOperation.copyDir(request);
    }

    @Override
    public List<FileOpTree> batchCopy(List<String> fileIds, List<String> dirIds, String targetDirId) throws Exception {
        FileOpRequest request = new FileOpRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        request.setTargetDirId(targetDirId);
        return this.batchCopy(request);
    }

    @Override
    public List<FileOpTree> batchCopy(FileOpRequest request) throws Exception {
        return copyOperation.copyMulti(request);
    }

    @Override
    public void move(String fileId) throws Exception {
        this.move(fileId, DMCConstants.EMPTY_UUID_STR);
    }

    @Override
    public void move(String fileId, String targetDirId) throws Exception {
        FileOpRequest request = new FileOpRequest();
        request.setFileId(fileId);
        request.setTargetDirId(targetDirId);
        moveOperation.move(request);
    }

    @Override
    public void moveDir(String dirId) throws Exception {
        this.moveDir(dirId, DMCConstants.EMPTY_UUID_STR);
    }

    @Override
    public void moveDir(String dirId, String targetDirId) throws Exception {
        FileOpRequest request = new FileOpRequest();
        request.setDirId(dirId);
        request.setTargetDirId(targetDirId);
        moveOperation.moveDir(request);
    }

    @Override
    public void batchMove(List<String> fileIds, List<String> dirIds, String targetDirId) throws Exception {
        FileOpRequest request = new FileOpRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        request.setTargetDirId(targetDirId);
        moveOperation.moveMulti(request);
    }

    @Override
    public void batchMove(FileOpRequest request) throws Exception {
        moveOperation.moveMulti(request);
    }

    @Override
    public void delete(String fileId) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileId(fileId);
        deleteOperation.deleteForce(request);
    }

    @Override
    public void deleteDir(String dirId) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setDirIds(Collections.singletonList(dirId));
        this.batchDelete(request);
    }

    @Override
    public void batchDelete(List<String> fileIds, List<String> dirIds) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        this.batchDelete(request);
    }

    @Override
    public void batchDelete(BatchRequest request) throws Exception {
        deleteOperation.deleteForceMulti(request);
    }

    @Override
    public FileInfo upload(File file) throws Exception {
        return this.upload(file, DMCConstants.EMPTY_UUID_STR);
    }

    @Override
    public FileInfo upload(File file, String dirId) throws Exception {
        return this.upload(file, null, dirId);
    }

    @Override
    public FileInfo upload(File file, String filename, String dirId) throws Exception {
        return this.upload(file, new FileInfo(filename, dirId));
    }

    @Override
    public FileInfo upload(File file, FileInfo fileInfo) throws Exception {
        UploadRequest request = new UploadRequest();
        request.setFile(file);
        request.setFileInfo(fileInfo);
        return this.upload(request);
    }

    @Override
    public FileInfo upload(byte[] bytes, String filename) throws Exception {
        return this.upload(bytes, filename, DMCConstants.EMPTY_UUID_STR);
    }

    @Override
    public FileInfo upload(byte[] bytes, String filename, String dirId) throws Exception {
        return this.upload(bytes, new FileInfo(filename, dirId));
    }

    @Override
    public FileInfo upload(byte[] bytes, FileInfo fileInfo) throws Exception {
        UploadRequest request = new UploadRequest();
        request.setBytes(bytes);
        request.setFileInfo(fileInfo);
        return this.upload(request);
    }

    @Override
    public FileInfo upload(InputStream input, String filename) throws Exception {
        return this.upload(input, filename, DMCConstants.EMPTY_UUID_STR);
    }

    @Override
    public FileInfo upload(InputStream input, String filename, String dirId) throws Exception {
        return this.upload(input, new FileInfo(filename, dirId));
    }

    @Override
    public FileInfo upload(InputStream input, FileInfo fileInfo) throws Exception {
        UploadRequest request = new UploadRequest();
        request.setInput(input);
        request.setFileInfo(fileInfo);
        return this.upload(request);
    }

    @Override
    public FileInfo upload(UploadRequest request) throws Exception {
        return uploadOperation.upload(request);
    }

    @Override
    public FileInfo coverUpload(String fileId, File file) throws Exception {
        UploadCoverRequest request = new UploadCoverRequest();
        request.setFileId(fileId);
        request.setFile(file);
        return uploadOperation.coverUpload(request);
    }

    @Override
    public FileInfo coverUpload(String fileId, byte[] bytes) throws Exception {
        FileInfo fileInfo = this.getFileInfo(fileId);
        if (fileInfo == null) {
            throw new DMCException(String.format("文件[%s]不存在", fileId));
        }
        UploadCoverRequest request = new UploadCoverRequest();
        request.setFileId(fileId);
        request.setFileInfo(fileInfo);
        request.setBytes(bytes);
        return uploadOperation.coverUpload(request);
    }

    @Override
    public FileInfo coverUpload(String fileId, InputStream input) throws Exception {
        FileInfo fileInfo = this.getFileInfo(fileId);
        if (fileInfo == null) {
            throw new DMCException(String.format("文件[%s]不存在", fileId));
        }
        UploadCoverRequest request = new UploadCoverRequest();
        request.setFileId(fileId);
        request.setFileInfo(fileInfo);
        request.setInput(input);
        return uploadOperation.coverUpload(request);
    }

    @Override
    public List<FileInfo> batchUpload(List<File> files) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setFiles(files);
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUpload(List<File> files, List<FileInfo> fileInfos) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setFiles(files);
        request.setFileInfos(fileInfos);
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUploadBytes(List<byte[]> bytes, List<String> filenames) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setBytes(bytes);
        request.setFileInfos(toFileInfo(filenames));
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUploadBytes(List<byte[]> bytes, Map<String, FileInfo> fileInfos) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setBytes(bytes);
        request.setFileInfos(toFileInfo(fileInfos));
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUploadStream(List<InputStream> inputs, List<String> filenames) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setInputs(inputs);
        request.setFileInfos(toFileInfo(filenames));
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUploadStream(List<InputStream> inputs, Map<String, FileInfo> fileInfos) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setInputs(inputs);
        request.setFileInfos(toFileInfo(fileInfos));
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUpload(UploadMultiRequest request) throws Exception {
        return uploadOperation.uploadMulti(request);
    }

    @Override
    public String multipartUpload(File file) throws Exception {
        return multipartUpload(file, DMCConstants.DEFAULT_PART_SIZE);
    }

    @Override
    public String multipartUpload(File file, int partSize) throws Exception {
        FileInfo fileInfo = new FileInfo();
        fileInfo.setFileName(file.getName());
        return multipartUpload(file, fileInfo, partSize);
    }

    @Override
    public String multipartUpload(File file, FileInfo fileInfo) throws Exception {
        return multipartUpload(file, fileInfo, DMCConstants.DEFAULT_PART_SIZE);
    }

    @Override
    public String multipartUpload(File file, FileInfo fileInfo, int partSize) throws Exception {
        MultipartUploadRequest request = new MultipartUploadRequest();
        request.setFile(file);
        request.setFileInfo(fileInfo);
        request.setFileLength(file.length());
        request.setPartSize(partSize);
        return multipartUpload(request);
    }

    @Override
    public String multipartUpload(InputStream input, FileInfo fileInfo) throws Exception {
        return multipartUpload(input, fileInfo, DMCConstants.DEFAULT_PART_SIZE);
    }

    @Override
    public String multipartUpload(InputStream input, FileInfo fileInfo, int partSize) throws Exception {
        MultipartUploadRequest request = new MultipartUploadRequest();
        request.setInput(input);
        request.setFileInfo(fileInfo);
        request.setFileLength(fileInfo.getSize());
        request.setPartSize(partSize);
        return multipartUpload(request);
    }

    @Override
    public String multipartUpload(MultipartUploadRequest request) throws Exception {
        return uploadOperation.multipartUpload(request);
    }

    @Override
    public byte[] download(String fileId) throws Exception {
        DownloadRequest request = new DownloadRequest();
        request.setFileId(fileId);
        return downloadOperation.download(request);
    }

    @Override
    public void download(String fileId, String filePath) throws Exception {
        String fileName = this.getFileInfo(fileId).getFileName();
        this.download(fileId, filePath, fileName);
    }

    @Override
    public void download(String fileId, String filePath, String fileName) throws Exception {
        DownloadRequest request = new DownloadRequest();
        request.setFileId(fileId);
        byte[] bytes = downloadOperation.download(request);
        FileCopyUtils.copy(bytes, FileUtils.newOutputStream(filePath, fileName));
    }

    @Override
    public void download(final String fileId, final OutputStream output) throws Exception {
        DownloadRequest request = new DownloadRequest();
        request.setFileId(fileId);
        byte[] bytes = downloadOperation.download(request);
        output.write(bytes);
    }

    @Override
    public byte[] batchDownload(List<String> fileIds, List<String> dirIds) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        return this.batchDownload(request);
    }

    @Override
    public void batchDownload(List<String> fileIds, List<String> dirIds, String filePath) throws Exception {
        String fileName = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMddHHmmss")) + ".zip";
        this.batchDownload(fileIds, dirIds, filePath, fileName);
    }

    @Override
    public void batchDownload(List<String> fileIds, List<String> dirIds, String filePath, String fileName) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        byte[] bytes = this.batchDownload(request);
        FileCopyUtils.copy(bytes, FileUtils.newOutputStream(filePath, fileName));
    }

    @Override
    public void batchDownload(List<String> fileIds, List<String> dirIds, OutputStream output) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        byte[] bytes = this.batchDownload(request);
        output.write(bytes);
    }

    @Override
    public byte[] batchDownload(BatchRequest request) throws Exception {
        return downloadOperation.downloadMulti(request);
    }

    @Override
    public List<ImageInfo> getImageInfo(String fileId) throws Exception {
        return getImageInfo(fileId, true);
    }

    @Override
    public List<ImageInfo> getImageInfo(String fileId, boolean resize) throws Exception {
        ImageRequest request = new ImageRequest();
        request.setFileId(fileId);
        request.setShrink(resize);
        return imageOperation.imageInfo(request);
    }

    @Override
    public ImageInfo resizeImageWidth(String fileId, int width) throws Exception {
        return resizeImage(fileId, width, 0, true);
    }

    @Override
    public ImageInfo resizeImageWidth(String fileId, int width, boolean share) throws Exception {
        return resizeImage(fileId, width, 0, share);
    }

    @Override
    public ImageInfo resizeImageHeight(String fileId, int height) throws Exception {
        return resizeImage(fileId, 0, height, true);
    }

    @Override
    public ImageInfo resizeImageHeight(String fileId, int height, boolean share) throws Exception {
        return resizeImage(fileId, 0, height, share);
    }

    @Override
    public ImageInfo resizeImage(String fileId, int width, int height) throws Exception {
        return resizeImage(fileId, width, height, true);
    }

    @Override
    public ImageInfo resizeImage(String fileId, int width, int height, boolean share) throws Exception {
        ImageRequest request = new ImageRequest();
        request.setFileId(fileId);
        request.setShrink(true);
        request.setWidth(width);
        request.setHeight(height);
        request.setShare(share);
        return imageOperation.imageShrink(request);
    }

    @Override
    public ImageUpload resizeImage(File image, FileInfo imageInfo) throws Exception {
        return this.resizeImage(image, imageInfo, 0, 0);
    }

    @Override
    public ImageUpload resizeImageWidth(File image, FileInfo imageInfo, int width) throws Exception {
        return this.resizeImage(image, imageInfo, width, 0);
    }

    @Override
    public ImageUpload resizeImageHeight(File image, FileInfo imageInfo, int height) throws Exception {
        return this.resizeImage(image, imageInfo, 0, height);
    }

    @Override
    public ImageUpload resizeImage(File image, FileInfo imageInfo, int width, int height) throws Exception {
        ImageRequest request = new ImageRequest();
        request.setFile(image);
        request.setFileInfo(imageInfo);
        request.setShrink(true);
        request.setWidth(width);
        request.setHeight(height);
        request.addHeader(DMCHeaders.HTTP_HEADER_API_ARG_KEY, URLEncoder.encode(JsonUtils.writeValue(imageInfo), "UTF-8"));
        return imageOperation.imageUpload(request);
    }

    @Override
    public RecycleBin remove(String fileId) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileId(fileId);
        return deleteOperation.delete(request);
    }

    @Override
    public RecycleBin removeDir(String dirId) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setDirIds(Collections.singletonList(dirId));
        return this.batchRemove(request).get(0);
    }

    @Override
    public List<RecycleBin> batchRemove(List<String> fileIds, List<String> dirIds) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        return this.batchRemove(request);
    }

    @Override
    public List<RecycleBin> batchRemove(BatchRequest request) throws Exception {
        return deleteOperation.deleteMulti(request);
    }

    @Override
    public void deleteRecycleBin(String recycleId) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setRecycleId(recycleId);
        recycleOperation.recycleDelete(request);
    }

    @Override
    public void batchDeleteRecycleBin(List<String> recycleIds) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setRecycleIds(recycleIds);
        recycleOperation.recycleDeleteMulti(request);
    }

    @Override
    public void clearRecycleBin() throws Exception {
        BatchRequest request = new BatchRequest();
        recycleOperation.recycleClear(request);
    }

    @Override
    public void restoreRecycleBin(String recycleId) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setRecycleId(recycleId);
        recycleOperation.recycleRestore(request);
    }

    @Override
    public void batchRestoreRecycleBin(List<String> recycleIds) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setRecycleIds(recycleIds);
        this.batchRestoreRecycleBin(request);
    }

    @Override
    public void batchRestoreRecycleBin(BatchRequest request) throws Exception {
        recycleOperation.recycleRestoreMulti(request);
    }

    @Override
    public void restore(String fileId) throws Exception {
        this.batchRestore(Collections.singletonList(fileId));
    }

    @Override
    public void batchRestore(List<String> fileIds) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileIds(fileIds);
        recycleOperation.restoreMulti(request);
    }

    @Override
    public ShareInfo share(String fileId) throws Exception {
        ShareRequest request = new ShareRequest();
        request.setFileId(fileId);
        return this.share(request);
    }

    @Override
    public ShareInfo share(String fileId, LocalDateTime expireDate) throws Exception {
        ShareRequest request = new ShareRequest();
        request.setFileId(fileId);
        request.setExpireDate(expireDate);
        return this.share(request);
    }

    @Override
    public ShareInfo share(String fileId, int allowCount) throws Exception {
        ShareRequest request = new ShareRequest();
        request.setFileId(fileId);
        request.setAllowCount(allowCount);
        return this.share(request);
    }

    @Override
    public ShareInfo share(String fileId, int allowCount, boolean deleteFile) throws Exception {
        ShareRequest request = new ShareRequest();
        request.setFileId(fileId);
        request.setAllowCount(allowCount);
        request.setDeleteFile(deleteFile);
        return this.share(request);
    }

    @Override
    public ShareInfo share(ShareRequest request) throws Exception {
        return shareOperation.shareMulti(request);
    }

    @Override
    public void deleteShare(String shareId) throws Exception {
        this.batchDeleteShare(Collections.singletonList(shareId));
    }

    @Override
    public void batchDeleteShare(List<String> shareIds) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setShareIds(shareIds);
        shareOperation.shareDeleteMulti(request);
    }

    @Override
    public DirInfo createDir(DirInfo dirInfo) throws Exception {
        DirRequest request = new DirRequest();
        request.setDirInfo(dirInfo);
        return baseOperation.addDirInfo(request);
    }

    @Override
    public DirInfo renameDir(String dirId, String dirName) throws Exception {
        DirRequest request = new DirRequest();
        DirInfo dirInfo = new DirInfo();
        dirInfo.setId(dirId);
        dirInfo.setName(dirName);
        request.setDirInfo(dirInfo);
        return baseOperation.renameDir(request);
    }

    @Override
    public DirInfo modDirInfo(DirInfo dirInfo) throws Exception {
        DirRequest request = new DirRequest();
        request.setDirInfo(dirInfo);
        return baseOperation.modDirInfo(request);
    }

    @Override
    public DirInfo getDirInfo(String dirId) throws Exception {
        DirRequest request = new DirRequest();
        request.setDirId(dirId);
        return baseOperation.getDirInfo(request);
    }

    @Override
    public DirInfo getDirInfoByName(String dirName) throws Exception {
        return getDirInfoByName(DMCConstants.EMPTY_UUID_STR, dirName);
    }

    @Override
    public DirInfo getDirInfoByName(String parentDirId, String dirName) throws Exception {
        DirRequest request = new DirRequest();
        DirInfo dirInfo = new DirInfo();
        dirInfo.setParentId(DMCUtils.getDirId(parentDirId));
        dirInfo.setName(dirName);
        request.setDirInfo(dirInfo);
        try {
            return baseOperation.getDirInfoByName(request);
        } catch (DMCException e) {
            if ("20404".equals(e.getErrorCode())) {
                // 临时兼容，新API V4.31- 2023-08-31上线
                return this.getDirInfoByNameV1(parentDirId, dirName);
            }
            throw e;
        }
    }

    @Override
    public FileInfo renameFile(String fileId, String fileName) throws Exception {
        FileInfoRequest request = new FileInfoRequest();
        FileInfo fileInfo = new FileInfo();
        fileInfo.setId(fileId);
        fileInfo.setFileName(fileName);
        request.setFileInfo(fileInfo);
        return baseOperation.renameFile(request);
    }

    @Override
    public FileInfo modFileInfo(FileInfo fileInfo) throws Exception {
        FileInfoRequest request = new FileInfoRequest();
        request.setFileInfo(fileInfo);
        return baseOperation.modFileInfo(request);
    }

    @Override
    public FileInfo modFileExpireDate(String fileId, LocalDateTime expireDate) throws Exception {
        FileInfo fileInfo = this.getFileInfo(fileId);
        fileInfo.setExpireDate(expireDate);
        return this.modFileInfo(fileInfo);
    }

    @Override
    public FileInfo modFileDirectoryId(String fileId, String dirId) throws Exception {
        FileInfo fileInfo = this.getFileInfo(fileId);
        fileInfo.setDirectoryId(dirId);
        return this.modFileInfo(fileInfo);
    }

    @Override
    public FileInfo getFileInfo(String fileId) throws Exception {
        FileInfoRequest request = new FileInfoRequest();
        request.setFileId(fileId);
        return baseOperation.getFileInfo(request);
    }

    @Override
    public FileInfo getFileInfoByName(String fileName) throws Exception {
        return getFileInfoByName(DMCConstants.EMPTY_UUID_STR, fileName);
    }

    @Override
    public FileInfo getFileInfoByName(String dirId, String fileName) throws Exception {
        FileInfoRequest request = new FileInfoRequest();
        FileInfo fileInfo = new FileInfo();
        fileInfo.setDirectoryId(dirId);
        fileInfo.setFileName(fileName);
        request.setFileInfo(fileInfo);
        return baseOperation.getFileInfoByName(request);
    }

    @Override
    public List<FileInfo> getFileInfos(String dirId) throws Exception {
        FileInfosRequest request = new FileInfosRequest();
        request.setDirId(dirId);
        return baseOperation.getFileInfoByDirId(request);
    }

    @Override
    public List<FileInfo> getFileInfos(List<String> fileIds) throws Exception {
        FileInfosRequest request = new FileInfosRequest();
        request.setFileIds(fileIds);
        return baseOperation.getFileInfoByIds(request);
    }

    @Override
    public String getToken() {
        return this.getToken(serviceClient.getConfig().getTenantId());
    }

    @Override
    public String getToken(String tenantId) {
        return serviceClient.getAuthCache().getToken(serviceClient.getConfig(), tenantId);
    }

    @Override
    public void switchTenantId() {
        this.switchTenantId(DMCConstants.DEFAULT_TENANT);
    }

    @Override
    public void switchTenantId(String tenantId) {
        TenantHolder.clearContext();
        serviceClient.setAuthCache(tenantId);
    }

    @Override
    public DMC setTenantId(String tenantId) {
        final String currentTenantId = serviceClient.getConfig().getTenantId();
        if (!DMCConstants.DEFAULT_TENANT.equals(currentTenantId)) {
            throw new DMCException(com.digiwin.dap.middleware.dmc.internal.DMCUtils.DMC_RESOURCE_MANAGER.getFormattedString("SwitchTenantIdDeny", currentTenantId));
        }
        TenantHolder.setContext(tenantId);
        return this;
    }

    @Override
    public void shutdown() {
        try {
            serviceClient.shutdown();
        } catch (Exception e) {
            logger.error("shutdown throw exception: ", e);
        }
    }

    @Override
    public String getConnectionPoolStats() {
        try {
            return serviceClient.getConnectionPoolStats();
        } catch (Exception ignored) {
        }
        return "";
    }
}
