package com.digiwin.dap.middleware.dmc.internal.client;

import com.digiwin.dap.middleware.dmc.internal.client.model.ShareV1Request;
import com.digiwin.dap.middleware.dmc.internal.client.model.UploadPartV1Request;
import com.digiwin.dap.middleware.dmc.internal.client.model.UploadV1Request;
import com.digiwin.dap.middleware.dmc.model.*;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

/**
 * 统一V1服务的调用方式，SDK内部使用
 *
 * @author fobgochod
 * @since 5.2.3.0
 */
public interface OSS {

    /**
     * 普通上传
     *
     * @param input    input 输入流
     * @param fileInfo 文件信息
     */
    FileInfo uploadV1(InputStream input, FileInfo fileInfo);

    FileInfo uploadV1(UploadV1Request request);

    /**
     * 覆盖上传
     *
     * @param fileId 文件ID
     * @param input  输入流
     */
    FileInfo uploadCoverV1(String fileId, InputStream input);

    FileInfo uploadCoverV1(UploadV1Request request);

    /**
     * 上传并分享
     *
     * @param input    input 输入流
     * @param fileInfo 文件信息
     */
    ShareInfo uploadShareV1(InputStream input, FileInfo fileInfo);

    ShareInfo uploadShareV1(UploadV1Request request);

    /**
     * 分段上传<br>
     * Step1：获取fileId
     */
    String claimFileIdV1(UploadPartV1Request request);

    /**
     * 分段上传<br>
     * Step2：分段上传
     */
    String uploadPartV1(UploadPartV1Request request);

    /**
     * 覆盖某一段文件流
     */
    String uploadPartCoverV1(UploadPartV1Request request);

    /**
     * 下载指定文件
     *
     * @param fileId 被下载的文件ID
     * @param output 输出流
     */
    void downloadV1(String fileId, OutputStream output);

    byte[] downloadV1(DownloadRequest request);

    /**
     * 分段下载
     */
    byte[] downloadPartV1(DownloadPartRequest request);

    /**
     * 删除文件到回收站
     *
     * @param fileId 被删除的文件ID
     */
    String removeV1(String fileId);

    String removeV1(BatchRequest request);

    /**
     * 删除目录到回收站
     *
     * @param dirId 被删除的目录ID
     */
    String removeDirV1(String dirId);

    String removeDirV1(BatchRequest request);

    /**
     * 批量删除文件、目录到回收站
     *
     * @param fileIds 被删除的文件ID集合
     * @param dirIds  被删除的目录ID集合
     */
    void batchRemoveV1(List<String> fileIds, List<String> dirIds);

    void batchRemoveV1(BatchRequest request);

    /**
     * 永久删除指定文件
     *
     * @param fileId 被删除的文件ID
     */
    void deleteV1(String fileId);

    void deleteV1(BatchRequest request);

    /**
     * 分享文件-永久分享
     *
     * @param fileId 文件ID
     * @return 分享地址集合
     */
    String shareV1(String fileId);

    /**
     * 分享文件-永久分享
     *
     * @param fileIds 文件ID集合
     * @return 分享地址集合
     */
    List<String> shareV1(List<String> fileIds);

    /**
     * 分享文件-限时分享
     *
     * @param fileIds   文件ID集合
     * @param expireDay 分享天数
     * @return 分享地址集合
     */
    List<String> shareV1(List<String> fileIds, long expireDay);

    List<String> shareV1(ShareV1Request request);

    /**
     * 删除分享记录
     *
     * @param shareId 分享记录ID
     */
    void deleteShareV1(String shareId);

    void deleteShareV1(ShareV1Request request);

    /**
     * 复制文件到一个指定目录
     *
     * @param fileId      被复制的文件ID
     * @param targetDirId 目标目录ID
     * @return 新文件ID
     */
    String copyV1(String fileId, String targetDirId);

    String copyV1(FileOpRequest request);

    /**
     * 复制目录到一个指定目录
     *
     * @param dirId       被复制的目录ID
     * @param targetDirId 目标目录ID
     * @return 新目录ID
     */
    String copyDirV1(String dirId, String targetDirId);

    String copyDirV1(FileOpRequest request);

    /**
     * 批量复制
     *
     * @param fileIds     被复制的文件ID集合
     * @param dirIds      被复制的目录ID集合
     * @param targetDirId 目标目录ID
     */
    void batchCopyV1(List<String> fileIds, List<String> dirIds, String targetDirId);

    void batchCopyV1(FileOpRequest request);

    /**
     * 剪切文件到一个指定目录
     *
     * @param fileId      被剪切的文件ID
     * @param targetDirId 目标目录ID
     */
    void moveV1(String fileId, String targetDirId);

    void moveV1(FileOpRequest request);

    /**
     * 剪切目录到一个指定目录
     *
     * @param dirId       被剪切的目录ID
     * @param targetDirId 目标目录ID
     */
    void moveDirV1(String dirId, String targetDirId);

    void moveDirV1(FileOpRequest request);

    /**
     * 批量剪切
     *
     * @param fileIds     被剪切的文件ID集合
     * @param dirIds      被剪切的目录ID集合
     * @param targetDirId 目标目录ID
     */
    void batchMoveV1(List<String> fileIds, List<String> dirIds, String targetDirId);

    void batchMoveV1(FileOpRequest request);

    /**********************************************************************
     *                                                                    *
     *                             目录基本信息操作                          *
     *                                                                    *
     **********************************************************************/

    /**
     * 创建目录
     *
     * @param dirName 目录名称
     */
    String createDirV1(String dirName);

    /**
     * 创建目录
     *
     * @param dirName  目录名称
     * @param parentId 父目录ID
     */
    String createDirV1(String dirName, String parentId);

    String createDirV1(DirRequest request);

    /**
     * 重命名目录
     *
     * @param dirId   目录ID
     * @param dirName 新目录名称
     */
    void renameDirV1(String dirId, String dirName);

    void renameDirV1(DirRequest request);

    /**
     * 获取目录信息
     *
     * @param dirId 目录ID
     */
    DirInfo getDirInfoV1(String dirId);

    DirInfo getDirInfoV1(DirRequest request);

    /**
     * 获取目录信息
     *
     * @param parentDirId 父目录ID
     * @param dirName     目录名称
     */
    DirInfo getDirInfoByNameV1(String parentDirId, String dirName);

    /**
     * 列出当前目录下的文件和子目录(不递归查询)
     *
     * @param dirId 目录ID
     */
    FolderInfo getDirContentV1(String dirId);

    FolderInfo getDirContentV1(DirRequest request);

    /**********************************************************************
     *                                                                    *
     *                             文件基本信息操作                          *
     *                                                                    *
     **********************************************************************/

    /**
     * 重命名文件
     *
     * @param fileId   文件ID
     * @param fileName 新文件名 带后缀名 <span>VanGogh.jpg</span>
     */
    void renameV1(String fileId, String fileName);

    void renameV1(FileInfoRequest request);

    /**
     * 获取文件信息
     *
     * @param fileId 文件ID
     */
    FileInfo getFileInfoV1(String fileId);

    FileInfo getFileInfoV1(FileInfoRequest request);


    /**
     * 登录
     *
     * @return userToken
     */
    String getToken(String username, String pwdhash, String tenantId);
}
