package com.digiwin.dap.middleware.dmc.internal.client.operation;

import com.digiwin.dap.middleware.dmc.HttpMethod;
import com.digiwin.dap.middleware.dmc.common.comm.RequestMessage;
import com.digiwin.dap.middleware.dmc.common.comm.ServiceClient;
import com.digiwin.dap.middleware.dmc.internal.DMCOperation;
import com.digiwin.dap.middleware.dmc.internal.DMCRequestMessageBuilder;
import com.digiwin.dap.middleware.dmc.internal.ResponseParsers;
import com.digiwin.dap.middleware.dmc.internal.client.parser.ResponseV1Parsers;
import com.digiwin.dap.middleware.dmc.model.DirInfo;
import com.digiwin.dap.middleware.dmc.model.DirRequest;
import com.digiwin.dap.middleware.dmc.model.FileOpRequest;
import com.digiwin.dap.middleware.dmc.model.FolderInfo;

/**
 * 目录信息相关操作
 *
 * @author fobgochod
 * @date 2023/8/7 16:25
 */
public class DirInfoV1Operation extends DMCOperation {

    public DirInfoV1Operation(ServiceClient client) {
        super(client);
    }

    /**
     * 创建目录<br>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/directorys
     */
    public String createDir(DirRequest original) throws Exception {
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getCreateDirUrlV1(original.getBucket()))
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setEntity(original.getEntity())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .build();
        return doOperation(request, ResponseV1Parsers.dirIdResponseParser);
    }

    /**
     * 更新目录名称<br>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}/rename/{dirName}
     */
    public void renameDir(DirRequest original) throws Exception {
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getRenameDirUrlV1(original.getBucket(), original.getDirId(), original.getDirName()))
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .build();
        doOperation(request, ResponseParsers.emptyResponseParser);
    }

    /**
     * 获取目录信息<br>
     * {@link HttpMethod#GET} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}
     */
    public DirInfo getDirInfo(DirRequest original) throws Exception {
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getDirInfoUrlV1(original.getBucket(), original.getDirId()))
                .setMethod(HttpMethod.GET)
                .setHeaders(original.getHeaders())
                .setOriginalRequest(original)
                .build();
        return doOperation(request, ResponseV1Parsers.dirInfoResponseParser);
    }

    /**
     * 获取目录信息<br>
     * {@link HttpMethod#GET} - /api/dmc/v1/buckets/{bucket}/directorys/{parentId}/{dirName}
     */
    public DirInfo getDirInfoByName(DirRequest original) throws Exception {
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getDirInfoGetByNameUrlV1(original.getDirInfo().getParentId(), original.getDirInfo().getName()))
                .setMethod(HttpMethod.GET)
                .setHeaders(original.getHeaders())
                .setOriginalRequest(original)
                .build();
        return doOperation(request, ResponseParsers.dirInfoResponseParser);
    }

    /**
     * 列出目录下的文件和子目录<br>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}/list
     */
    public FolderInfo getDirContent(DirRequest original) throws Exception {
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getDirContentUrlV1(original.getBucket(), original.getDirId()))
                .setMethod(HttpMethod.GET)
                .setHeaders(original.getHeaders())
                .setOriginalRequest(original)
                .build();
        return doOperation(request, ResponseV1Parsers.dirContentResponseParser);
    }

    /**
     * 复制目录<br>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}/copyto/{targetDirId}
     */
    public String copyDir(FileOpRequest original) throws Exception {
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getDirCopyUrlV1(original.getBucket(), original.getDirId(), original.getTargetDirId()))
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .build();
        return doOperation(request, ResponseV1Parsers.dirIdResponseParser);
    }

    /**
     * 移动目录<br>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}/moveto/{targetDirId}
     */
    public void moveDir(FileOpRequest original) throws Exception {
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getDirMoveUrlV1(original.getBucket(), original.getDirId(), original.getTargetDirId()))
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .build();
        doOperation(request, ResponseParsers.emptyResponseParser);
    }
}
