package com.digiwin.dap.middleware.dmc.internal.client.parser;

import com.digiwin.dap.middleware.dmc.common.comm.ResponseMessage;
import com.digiwin.dap.middleware.dmc.common.parser.ResponseParseException;
import com.digiwin.dap.middleware.dmc.common.parser.ResponseParser;
import com.digiwin.dap.middleware.dmc.internal.DMCUtils;
import com.digiwin.dap.middleware.dmc.model.DirInfo;
import com.digiwin.dap.middleware.dmc.model.FileInfo;
import com.digiwin.dap.middleware.dmc.model.FolderInfo;
import com.digiwin.dap.middleware.dmc.model.ShareInfo;
import com.digiwin.dap.middleware.util.JsonUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;

import java.util.List;
import java.util.Map;

/**
 * A collection of parsers that parse HTTP responses into corresponding human-readable results.
 *
 * @author fobgochod
 * @date 2023/8/7 10:59
 */
public final class ResponseV1Parsers {

    public static final FileInfoResponseParser fileInfoResponseParser = new FileInfoResponseParser();
    public static final FileIdResponseParsDirDirer fileIdResponseParser = new FileIdResponseParsDirDirer();
    public static final DirInfoResponseParser dirInfoResponseParser = new DirInfoResponseParser();
    public static final DirContentResponseParser dirContentResponseParser = new DirContentResponseParser();
    public static final DirIdResponseParsDirDirer dirIdResponseParser = new DirIdResponseParsDirDirer();
    public static final ShareInfoResponseParser shareInfoResponseParser = new ShareInfoResponseParser();
    public static final ShareUrlResponseParser shareUrlResponseParser = new ShareUrlResponseParser();
    public static final RecycleBinIdResponseParser recycleBinIdResponseParser = new RecycleBinIdResponseParser();

    private static final ObjectMapper objectMapper = JsonUtils.createObjectMapper();

    public static final class FileInfoResponseParser implements ResponseParser<FileInfo> {
        @Override
        public FileInfo parse(ResponseMessage response) throws ResponseParseException {
            try {
                return objectMapper.readValue(response.getContent(), FileInfo.class);
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class FileIdResponseParsDirDirer implements ResponseParser<String> {
        @Override
        public String parse(ResponseMessage response) throws ResponseParseException {
            try {
                Map<String, String> result = objectMapper.readValue(response.getContent(), new TypeReference<Map<String, String>>() {
                });
                return result.get("id");
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }


    public static final class DirInfoResponseParser implements ResponseParser<DirInfo> {
        @Override
        public DirInfo parse(ResponseMessage response) throws ResponseParseException {
            try {
                return objectMapper.readValue(response.getContent(), DirInfo.class);
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class DirContentResponseParser implements ResponseParser<FolderInfo> {
        @Override
        public FolderInfo parse(ResponseMessage response) throws ResponseParseException {
            try {
                return objectMapper.readValue(response.getContent(), FolderInfo.class);
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class DirIdResponseParsDirDirer implements ResponseParser<String> {
        @Override
        public String parse(ResponseMessage response) throws ResponseParseException {
            try {
                Map<String, String> result = objectMapper.readValue(response.getContent(), new TypeReference<Map<String, String>>() {
                });
                return result.get("dirId");
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class ShareInfoResponseParser implements ResponseParser<ShareInfo> {
        @Override
        public ShareInfo parse(ResponseMessage response) throws ResponseParseException {
            try {
                return objectMapper.readValue(response.getContent(), ShareInfo.class);
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class ShareUrlResponseParser implements ResponseParser<List<String>> {
        @Override
        public List<String> parse(ResponseMessage response) throws ResponseParseException {
            try {
                return objectMapper.readValue(response.getContent(), new TypeReference<List<String>>() {
                });
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class RecycleBinIdResponseParser implements ResponseParser<String> {
        @Override
        public String parse(ResponseMessage response) throws ResponseParseException {
            try {
                Map<String, String> result = objectMapper.readValue(response.getContent(), new TypeReference<Map<String, String>>() {
                });
                return result.get("recycleBinId");
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }
}
