package com.digiwin.dmc.sdk.service.upload;

import com.digiwin.dap.middleware.dmc.common.context.TenantId;
import com.digiwin.dap.middleware.dmc.internal.client.model.UploadPartV1Request;
import com.digiwin.dmc.sdk.config.NetworkOptions;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.util.ArgumentUtils;
import com.digiwin.dmc.sdk.util.ExecutorUtil;
import com.digiwin.dmc.sdk.util.HttpUtils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Map;

/**
 * 覆盖分段上传
 * /api/dmc/v1/buckets/{bucket}/files/{fileId}/{from}/{to}/{total}/cover
 *
 * @date 2022/6/7 14:12
 */
public class CoverSegmentDocumentUploader implements ICoverSegmentDocumentUploader {

    private String bucketName;
    private String fileId;
    private String driveToken;
    private TenantId[] tenantId;
    private int bufferSize = NetworkOptions.Default.getPackageSize();
    private InputStream inputStream;
    private IUpLoadCallbackable progressChangedCallback;
    private IUpLoadCallbackable completeCallback;
    private IUpLoadCallbackable errorCallback;
    private UploadProcessState state = UploadProcessState.NotStarted;
    private UploadProgressEventArgs eventArgs = new UploadProgressEventArgs();

    private CoverSegmentDocumentUploader(String bucketName, String fileId, String driveToken, TenantId... tenantId) {
        this.bucketName = bucketName;
        this.fileId = fileId;
        this.driveToken = driveToken;
        this.tenantId = tenantId;
    }

    public CoverSegmentDocumentUploader(String bucketName, String localPath, String fileId, String driveToken, TenantId... tenantId) {
        this(bucketName, fileId, driveToken, tenantId);
        try {
            this.inputStream = Files.newInputStream(Paths.get(localPath));
            eventArgs.setTotalBytes(inputStream.available());
        } catch (IOException e) {
            throw new OperateException(e);
        }
        eventArgs.setPercentage(0);
        eventArgs.setCompletedBytes(0);
    }

    public CoverSegmentDocumentUploader(String bucketName, byte[] bytes, String fileId, String driveToken, TenantId... tenantId) {
        this(bucketName, fileId, driveToken, tenantId);
        try {
            this.inputStream = new ByteArrayInputStream(bytes);
            eventArgs.setTotalBytes(inputStream.available());
        } catch (IOException e) {
            throw new OperateException(e);
        }
        eventArgs.setPercentage(0);
        eventArgs.setCompletedBytes(0);
    }

    /**
     * 异步上传Stream中的文件内容
     *
     * @return 上传文件Id
     */
    @Override
    public ICoverSegmentDocumentUploader beginUpload() {
        if (state != UploadProcessState.NotStarted) {
            throw new OperateException("上传已开始，无法再次启动下载，请开启新的上传");
        }
        bucketName = ArgumentUtils.getBucketName(bucketName);

        this.continueUpload();

        Map<String, String> headers = HttpUtils.setHeader(driveToken, tenantId);

        Runnable runnable = () -> {
            int totalBytes = eventArgs.getTotalBytes();
            int partCount = (int) Math.ceil((double) totalBytes / bufferSize);

            try {
                for (int i = 0; i < partCount; i++) {
                    int startPos = i * bufferSize;
                    int curPartSize = (i + 1 == partCount) ? (totalBytes - startPos) : bufferSize;

                    if (state == UploadProcessState.Stopped) {
                        break;
                    } else if (state == UploadProcessState.Paused) {
                        Thread.sleep(1000 * 60 * 60 * 24);
                        if (state == UploadProcessState.Paused) {
                            state = UploadProcessState.Timeout;
                            break;
                        }
                    } else {
                        byte[] buffer = new byte[curPartSize];
                        inputStream.read(buffer);
                        this.uploadSegFile(bucketName, headers, startPos, curPartSize, buffer);
                        eventArgs.setCompletedBytes(startPos + curPartSize);
                        eventArgs.setPercentage(Math.round(eventArgs.getCompletedBytes() * 10000.0 / totalBytes) / 10000.0);
                        if (progressChangedCallback != null) {
                            progressChangedCallback.callback(eventArgs);
                        }

                        if ((partCount - i == 1) && completeCallback != null) {
                            eventArgs.setFileId(fileId);
                            completeCallback.callback(eventArgs);
                        }
                    }
                }
            } catch (Exception e) {
                if (errorCallback != null) {
                    errorCallback.callback(eventArgs);
                }
                throw new OperateException(e);
            }

            /// ///

            // byte[] buffer = new byte[bufferSize];
            // int bytesReaded = 0;
            // int totalBytes = eventArgs.getTotalBytes();
            //
            // while (bytesReaded < totalBytes) {
            //     try {
            //         bytesReaded = eventArgs.getCompletedBytes();
            //         if (state == UploadProcessState.Stopped) {
            //             break;
            //         } else if (state == UploadProcessState.Paused) {
            //             Thread.sleep(1000 * 60 * 60 * 24);
            //             if (state == UploadProcessState.Paused) {
            //                 state = UploadProcessState.Timeout;
            //                 break;
            //             }
            //         } else {
            //             int currentReadLen = inputStream.read(buffer, 0, bufferSize);
            //             if (currentReadLen == -1) {
            //                 break;
            //             }
            //             if (bytesReaded + bufferSize >= totalBytes) {
            //                 currentReadLen = totalBytes - bytesReaded;
            //                 this.uploadSegFile(bucketName, headers, bytesReaded, currentReadLen, buffer);
            //                 eventArgs.setCompletedBytes(totalBytes);
            //                 eventArgs.setPercentage(1);
            //                 if (progressChangedCallback != null) {
            //                     progressChangedCallback.callback(eventArgs);
            //                 }
            //                 if (completeCallback != null) {
            //                     eventArgs.setFileId(fileId);
            //                     completeCallback.callback(eventArgs);
            //                 }
            //                 return;
            //             } else {
            //                 this.uploadSegFile(bucketName, headers, bytesReaded, currentReadLen, buffer);
            //                 eventArgs.setCompletedBytes(bytesReaded + currentReadLen);
            //                 eventArgs.setPercentage((double) eventArgs.getCompletedBytes() / (double) eventArgs.getTotalBytes());
            //                 if (progressChangedCallback != null) {
            //                     progressChangedCallback.callback(eventArgs);
            //                 }
            //             }
            //         }
            //     } catch (Exception e) {
            //         if (errorCallback != null) {
            //             errorCallback.callback(eventArgs);
            //         }
            //         throw new OperateException(e);
            //     }
            // }
        };
        ExecutorUtil.newExecutor().execute(runnable);
        return this;

    }

    public void uploadSegFile(String bucketName, Map<String, String> headers, int from, int count, byte[] buf) {
        UploadPartV1Request request = new UploadPartV1Request();
        request.setBucket(bucketName);
        request.setHeaders(headers);
        request.setFileId(fileId);
        request.setParts(buf);
        request.setFrom(from);
        request.setTo(from + count - 1);
        request.setTotal(this.eventArgs.getTotalBytes());
        ServerSetting.internal().uploadPartCoverV1(request);
    }

    /**
     * 设置响应进度
     *
     * @param callbackable
     */
    @Override
    public ICoverSegmentDocumentUploader onProgressChanged(IUpLoadCallbackable callbackable) {
        this.progressChangedCallback = callbackable;
        return this;
    }

    /**
     * 设置完成
     *
     * @param callbackable
     */
    @Override
    public ICoverSegmentDocumentUploader onCompleted(IUpLoadCallbackable callbackable) {
        this.completeCallback = callbackable;
        return this;
    }

    @Override
    public ICoverSegmentDocumentUploader onError(IUpLoadCallbackable callbackable) {
        this.errorCallback = callbackable;
        return this;
    }

    /**
     * 暂停上传
     */
    @Override
    public void pauseUpload() {
        state = UploadProcessState.Paused;
    }

    /**
     * 继续上传
     */
    @Override
    public void continueUpload() {
        state = UploadProcessState.Uploading;
    }

    /**
     * 停止上传
     */
    @Override
    public void stopUpload() {
        state = UploadProcessState.Stopped;
    }

}
