package com.digiwin.dmc.sdk.util;

import com.digiwin.dap.middleware.dmc.common.utils.IOUtils;
import com.digiwin.dmc.sdk.exception.OperateException;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

/**
 * http请求封装
 *
 * @author chenxs
 */
public class HttpRequestUtil {

    /**
     * 上传文件
     *
     * @param url         url地址
     * @param heads       头部
     * @param inputStream 文件输入流
     */
    public static <T> T uploadFile(String url, Map<String, String> heads, InputStream inputStream, Class<T> entityClass) {
        OutputStream out = null;
        DataInputStream in = null;
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection conn = (HttpURLConnection) realUrl.openConnection();
            setHeader(heads, conn);
            conn.setRequestProperty("Content-Type", "application/octet-stream");
            // 设置为POST情
            conn.setRequestMethod("POST");
            // 发送POST请求必须设置如下两行
            conn.setDoOutput(true);
            conn.setDoInput(true);
            conn.setUseCaches(false);

            out = new DataOutputStream(conn.getOutputStream());

            // 数据输入流,用于读取文件数据
            in = new DataInputStream(inputStream);
            byte[] bufferOut = new byte[1024];
            int bytes = 0;
            // 每次读1KB数据,并且将文件数据写入到输出流中
            while ((bytes = in.read(bufferOut)) != -1) {
                out.write(bufferOut, 0, bytes);
            }
            out.flush();
            //2获取返回的数据
            String result = getBody(conn);
            if (result.length() > 0) {
                return ObjectMapperUtil.readValue(result, entityClass);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            closeStream(out, in);
        }
        return null;
    }

    /**
     * 分段上传
     *
     * @param url
     * @param headers
     * @param count
     * @param buf
     */
    public static void uploadSegFile(String url, Map<String, String> headers, int count, byte[] buf) {
        OutputStream out = null;
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection conn = (HttpURLConnection) realUrl.openConnection();
            setHeader(headers, conn);
            conn.setRequestProperty("Content-Type", "application/octet-stream");
            // 设置为POST情
            conn.setRequestMethod("POST");
            // 发送POST请求必须设置如下两行
            conn.setDoOutput(true);
            conn.setDoInput(true);
            conn.setUseCaches(false);

            out = new DataOutputStream(conn.getOutputStream());

            out.write(buf, 0, count);
            out.flush();
            out.close();
            //2获取返回的数据
            String result = getBody(conn);
            conn.disconnect();


        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            try {
                if (out != null) {
                    out.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 获取文件
     *
     * @param url          url地址
     * @param heads        头部
     * @param outputStream 文件输出流
     */
    public static OutputStream downloadFile(String url, Map<String, String> heads, OutputStream outputStream) {

        InputStream is = null;
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection con = (HttpURLConnection) realUrl.openConnection();
            setHeader(heads, con);
            // 设置为GET情
            con.setRequestMethod("GET");
            // 建立实际的连接
            con.connect();

            errorHandler(con);

            is = con.getInputStream();
            byte[] buf = new byte[1024];
            int count;
            while ((count = is.read(buf)) != -1) {
                outputStream.write(buf, 0, count);
            }
            outputStream.flush();
            return outputStream;
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException ex) {
                ex.printStackTrace();
            }
        }
    }

    private static void errorHandler(HttpURLConnection con) throws IOException {
        int code = con.getResponseCode();
        if (code != 200 && code != 201) {
            String error = IOUtils.readStreamAsString(con.getErrorStream(), StandardCharsets.UTF_8.name());
            if (error.length() > 0) {
                Map map = ObjectMapperUtil.readValue(error, HashMap.class);
                throw new OperateException(String.format("status code: %s ,detail: %s", code, map.get("message")));
            } else {
                throw new OperateException(String.format("status code: %s ,detail: %s", code, error));
            }
        }
    }

    /**
     * 分段下载文件
     *
     * @param url          url地址
     * @param headers      头部
     * @param chunk        每包传输的数据量，单位字节
     * @param outputStream 文件保存路径
     * @return
     */
    public static OutputStream downloadSegFile(String url, Map<String, String> headers, int chunk, OutputStream outputStream) {
        InputStream is = null;
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection con = (HttpURLConnection) realUrl.openConnection();
            setHeader(headers, con);
            // 设置为GET情
            con.setRequestMethod("GET");
            // 建立实际的连接
            con.connect();

            errorHandler(con);

            is = con.getInputStream();
            writeFileOutputStream(chunk, is, outputStream);
            outputStream.flush();
            return outputStream;
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException ex) {
                ex.printStackTrace();
            }
        }
    }

    private static void writeFileOutputStream(int chunk, InputStream inStream, OutputStream outputStream) throws Exception {
        byte[] buffer = new byte[chunk];
        int len = 0;
        while ((len = inStream.read(buffer)) != -1) {
            outputStream.write(buffer, 0, len);
        }
        inStream.close();
    }

    /**
     * 向指定URL发送GET方法的请求
     *
     * @param url 发送请求的URL
     * @return URL 所代表远程资源的响应结果
     */
    public static <T> T get(String url, Map<String, String> heads, Class<T> entityClass) {
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection connection = (HttpURLConnection) realUrl.openConnection();
            setHeader(heads, connection);
            // 设置为GET情
            connection.setRequestMethod("GET");
            // 建立实际的连接
            connection.connect();
            //如果返回length
            String result = getBody(connection);
            connection.disconnect();
            if (result.length() > 0) {
                return ObjectMapperUtil.readValue(result, entityClass);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return null;
    }


    public static byte[] getByte(String url, Map<String, String> heads) {
        InputStream is = null;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection con = (HttpURLConnection) realUrl.openConnection();
            setHeader(heads, con);
            // 设置为GET情
            con.setRequestMethod("GET");
            // 建立实际的连接
            con.connect();
            is = con.getInputStream();
            //<2>创建缓存大小
            byte[] buffer = new byte[1024];
            //每次读取到内容的长度
            int len;
            //<3>开始读取输入流中的内容
            //当等于-1说明没有数据可以读取了
            while ((len = is.read(buffer)) != -1) {
                //把读取到的内容写到输出流中
                baos.write(buffer, 0, len);
            }
            return baos.toByteArray();
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException ex) {
                ex.printStackTrace();
            }
        }
    }

    /**
     * 向指定 URL 发送POST方法的请求
     *
     * @param url  发送请求的 URL
     * @param json 数据
     * @return 所代表远程资源的响应结果
     */
    public static <T> T postJson(String url, String json, Map<String, String> heads, Class<T> entityClass) throws IOException {
        OutputStreamWriter out = null;
        String result;
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection conn = (HttpURLConnection) realUrl.openConnection();
            setHeader(heads, conn);
            // 设置为POST情
            conn.setRequestMethod("POST");
            // 发送POST请求必须设置如下两行
            conn.setDoOutput(true);
            conn.setDoInput(true);

            //1.获取URLConnection对象对应的输出流,把json数据写入到请求中
            out = new OutputStreamWriter(conn.getOutputStream(), StandardCharsets.UTF_8);
            out.write(json);
            // flush输出流的缓冲
            out.flush();
            //2获取返回的数据
            result = getBody(conn);
            conn.disconnect();
        }
        //使用finally块来关闭输出流、输入流
        finally {
            try {
                if (out != null) {
                    out.close();
                }
            } catch (IOException ex) {
                ex.printStackTrace();
            }
        }
        if (result.length() > 0) {
            return ObjectMapperUtil.readValue(result, entityClass);
        }
        return null;
    }

    /**
     * 向指定URL发送DELETE方法的请求
     *
     * @param url 发送请求的URL
     * @return URL 所代表远程资源的响应结果
     */
    public static <T> T delete(String url, Map<String, String> heads, Class<T> entityClass) {
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection connection = (HttpURLConnection) realUrl.openConnection();
            setHeader(heads, connection);
            // 设置为GET情
            connection.setRequestMethod("DELETE");
            // 建立实际的连接
            connection.connect();
            //如果返回length
            String result = getBody(connection);
            connection.disconnect();
            if (result != null && result.length() > 0) {
                return ObjectMapperUtil.readValue(result, entityClass);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return null;
    }

    /**
     * 向指定URL发送DELETE方法的请求
     *
     * @param url 发送请求的URL
     * @return URL 所代表远程资源的响应结果
     */
    public static <T> T deleteJson(String url, String json, Map<String, String> heads, Class<T> entityClass) {
        OutputStreamWriter out = null;
        String result;
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection conn = (HttpURLConnection) realUrl.openConnection();
            setHeader(heads, conn);
            conn.setRequestMethod("DELETE");
            // 发送POST请求必须设置如下两行
            conn.setDoOutput(true);
            conn.setDoInput(true);

            //1.获取URLConnection对象对应的输出流,把json数据写入到请求中
            out = new OutputStreamWriter(conn.getOutputStream(), "UTF-8");
            out.write(json);
            // flush输出流的缓冲
            out.flush();
            //2获取返回的数据
            result = getBody(conn);
            conn.disconnect();

            if (result.length() > 0) {
                return ObjectMapperUtil.readValue(result, entityClass);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return null;
    }

    /**
     * 向指定 URL 发送PUT方法的请求
     *
     * @param url  发送请求的 URL
     * @param json 数据
     * @return 所代表远程资源的响应结果
     */
    public static <T> T put(String url, String json, Map<String, String> heads, Class<T> entityClass) throws IOException {
        OutputStreamWriter out = null;
        String result;
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection conn = (HttpURLConnection) realUrl.openConnection();
            setHeader(heads, conn);
            // 设置为POST情
            conn.setRequestMethod("PUT");
            // 发送POST请求必须设置如下两行
            conn.setDoOutput(true);
            conn.setDoInput(true);

            //1.获取URLConnection对象对应的输出流,把json数据写入到请求中
            out = new OutputStreamWriter(conn.getOutputStream(), StandardCharsets.UTF_8);
            out.write(json);
            // flush输出流的缓冲
            out.flush();
            //2获取返回的数据
            result = getBody(conn);
            conn.disconnect();
        }
        //使用finally块来关闭输出流、输入流
        finally {
            try {
                if (out != null) {
                    out.close();
                }
            } catch (IOException ex) {
                ex.printStackTrace();
            }
        }
        if (result.length() > 0) {
            return ObjectMapperUtil.readValue(result, entityClass);
        }
        return null;
    }

    private static void closeStream(OutputStream os, InputStream is) {
        try {
            if (os != null) {
                os.close();
            }
            if (is != null) {
                is.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private static void setHeader(Map<String, String> headers, URLConnection conn) {
        // 设置通用的请求属性
        conn.setRequestProperty("accept", "*/*");
        conn.setRequestProperty("Content-Type", "application/json; charset=utf-8");
        conn.setRequestProperty("connection", "keep-alive");
        conn.setRequestProperty("user-agent", "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1;SV1)");
        //设置头部参数
        for (Map.Entry<String, String> item : headers.entrySet()) {
            if (item.getValue() != null) {
                conn.addRequestProperty(item.getKey(), item.getValue());
            }
        }
    }

    private static String getBody(HttpURLConnection connection) {
        StringBuffer resultBuffer;
        // 定义 BufferedReader输入流来读取URL的响应
        BufferedReader in = null;
        try {
            resultBuffer = new StringBuffer();
            int code = connection.getResponseCode();
            if (code == 200) {
                in = new BufferedReader(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
            } else {
                in = new BufferedReader(new InputStreamReader(connection.getErrorStream(), StandardCharsets.UTF_8));
            }
            String temp;
            while ((temp = in.readLine()) != null) {
                resultBuffer.append(temp);
            }
            if (code != 200) {
                //处理结果
                if (resultBuffer.length() > 0) {
                    Map map = ObjectMapperUtil.readValue(resultBuffer.toString(), HashMap.class);
                    throw new OperateException(String.format("status code: %s ,detail: %s", code, map.get("message")));
                } else {
                    throw new OperateException(String.format("status code: %s ,detail: %s", code, resultBuffer.toString()));
                }
            }
            return resultBuffer.toString();
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            try {
                if (in != null) {
                    in.close();
                }
            } catch (Exception e2) {
                e2.printStackTrace();
            }
        }
    }

    public static <T> T postJson(String url, String json, Class<T> entityClass) throws IOException {
        OutputStreamWriter out = null;
        String result;
        try {
            URL realUrl = new URL(url);
            // 打开和URL之间的连接
            HttpURLConnection conn = (HttpURLConnection) realUrl.openConnection();
            conn.setRequestProperty("accept", "*/*");
            conn.setRequestProperty("Content-Type", "application/json");
            conn.setRequestProperty("connection", "keep-alive");
            conn.setRequestProperty("user-agent", "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1;SV1)");
            // 设置为POST情
            conn.setRequestMethod("POST");
            // 发送POST请求必须设置如下两行
            conn.setDoOutput(true);
            conn.setDoInput(true);

            //1.获取URLConnection对象对应的输出流,把json数据写入到请求中
            out = new OutputStreamWriter(conn.getOutputStream(), StandardCharsets.UTF_8);
            out.write(json);
            // flush输出流的缓冲
            out.flush();
            //2获取返回的数据
            result = getBody(conn);
        }
        //使用finally块来关闭输出流、输入流
        finally {
            try {
                if (out != null) {
                    out.close();
                }
            } catch (IOException ex) {
                ex.printStackTrace();
            }
        }
        if (result.length() > 0) {
            return ObjectMapperUtil.readValue(result, entityClass);
        }
        return null;
    }
}
