package com.digiwin.dmc.sdk.service.download;

import com.digiwin.dap.middleware.dmc.common.context.TenantId;
import com.digiwin.dap.middleware.dmc.model.DownloadPartRequest;
import com.digiwin.dmc.sdk.config.NetworkOptions;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.util.ArgumentUtils;
import com.digiwin.dmc.sdk.util.ExecutorUtil;
import com.digiwin.dmc.sdk.util.HttpUtils;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Map;
import java.util.concurrent.locks.LockSupport;

/**
 * 分段下载
 * /api/dmc/v1/buckets/{bucket}/files/{fileId}/{position}/{length}
 *
 * @author chenzwd
 * @date 2018-06-30 16:03
 */
public class SegmentSegmentDocumentDownloader implements ISegmentDocumentDownloader {

    private final String discardTenantId;
    private final String fileId;
    private final String driveToken;
    private final int bufferSize = NetworkOptions.Default.getPackageSize();
    private final DownloadProgressEventArgs eventArgs = new DownloadProgressEventArgs();
    private String bucketName;
    private TenantId[] tenantId;
    private OutputStream outputStream;
    private Thread thread;
    private volatile DownloadProcessState state = DownloadProcessState.NotStarted;
    private IDownLoadCallbackable completeCallback;
    private IDownLoadCallbackable progressChangedCallback;

    private SegmentSegmentDocumentDownloader(String discardTenantId, String bucketName, String fileId, String driveToken, TenantId... tenantId) {
        this.discardTenantId = discardTenantId;
        this.bucketName = bucketName;
        this.fileId = fileId;
        this.driveToken = driveToken;
        this.tenantId = tenantId;
    }

    public SegmentSegmentDocumentDownloader(String discardTenantId, String bucketName, String fileId, String saveToPath, int fileSize, String driveToken, TenantId... tenantId) {
        this(discardTenantId, bucketName, fileId, driveToken, tenantId);
        try {
            this.outputStream = Files.newOutputStream(Paths.get(saveToPath));
        } catch (IOException e) {
            throw new OperateException(e);
        }
        this.eventArgs.setTotalBytes(fileSize);
    }

    /**
     * 异步下载文件内容
     */
    @Override
    public ISegmentDocumentDownloader beginDownload() {
        if (state != DownloadProcessState.NotStarted) {
            throw new OperateException("下载已开始，无法再次启动下载，请开启新的下载");
        }
        bucketName = ArgumentUtils.getBucketName(bucketName);
        Map<String, String> headers = HttpUtils.setHeader(driveToken, discardTenantId, tenantId);

        this.continueDownload();

        Runnable runnable = () -> {
            this.multipartDownload(bucketName, headers);
        };
        thread = ExecutorUtil.THREAD_FACTORY.newThread(runnable);
        thread.start();
        return this;
    }


    private void multipartDownload(String bucketName, Map<String, String> headers) {
        try {
            int totalBytes = eventArgs.getTotalBytes();
            int downloadBytes = 0;
            while (downloadBytes < totalBytes) {
                if (state == DownloadProcessState.Stopped) {
                    return;
                } else if (state == DownloadProcessState.Paused) {
                    LockSupport.park(thread);
                } else {
                    //from的位置是二进制数组的下标，所有需要把长度-1
                    int from = downloadBytes;
                    if (downloadBytes + bufferSize >= totalBytes) {
                        this.download(bucketName, headers, from, totalBytes - downloadBytes);
                        downloadBytes = totalBytes;
                        eventArgs.setCompletedBytes(totalBytes);
                        eventArgs.setPercentage(1);
                        if (progressChangedCallback != null) {
                            progressChangedCallback.callback(eventArgs);
                        }
                        if (completeCallback != null) {
                            completeCallback.callback(eventArgs);
                        }
                    } else {
                        this.download(bucketName, headers, from, bufferSize);
                        downloadBytes += bufferSize;
                        eventArgs.setCompletedBytes(downloadBytes);
                        eventArgs.setPercentage(Math.round(eventArgs.getCompletedBytes() * 10000.0 / totalBytes) / 10000.0);
                        if (progressChangedCallback != null) {
                            progressChangedCallback.callback(eventArgs);
                        }
                    }
                }
            }
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    private void download(String bucketName, Map<String, String> headers, int startPos, int length) {
        DownloadPartRequest request = new DownloadPartRequest();
        request.setBucket(bucketName);
        request.setHeaders(headers);
        request.setFileId(fileId);
        request.setOutput(outputStream);
        request.setPosition(startPos);
        request.setLength(length);
        ServerSetting.internal().downloadPartV1(request);
    }

    @Override
    public ISegmentDocumentDownloader onProgressChanged(IDownLoadCallbackable callbackable) {
        this.progressChangedCallback = callbackable;
        return this;
    }

    @Override
    public ISegmentDocumentDownloader onCompleted(IDownLoadCallbackable callbackable) {
        this.completeCallback = callbackable;
        return this;
    }

    @Override
    public void pauseDownload() {
        state = DownloadProcessState.Paused;
    }

    @Override
    public void continueDownload() {
        if (state == DownloadProcessState.Paused) {
            LockSupport.unpark(thread);
        }
        state = DownloadProcessState.Downloading;
    }

    @Override
    public void stopDownload() {
        state = DownloadProcessState.Stopped;
    }
}
