package com.digiwin.dap.middleware.dmc.common.comm;

import com.digiwin.dap.middleware.dmc.ClientConfiguration;
import com.digiwin.dap.middleware.dmc.common.app.AppCache;
import com.digiwin.dap.middleware.dmc.common.auth.AuthCache;
import com.digiwin.dap.middleware.dmc.common.context.IamUserTokenHolder;
import com.digiwin.dap.middleware.dmc.common.context.TenantHolder;
import com.digiwin.dap.middleware.dmc.common.utils.ExceptionFactory;
import com.digiwin.dap.middleware.dmc.common.utils.HttpHeaders;
import com.digiwin.dap.middleware.dmc.common.utils.HttpUtil;
import com.digiwin.dap.middleware.dmc.common.utils.StreamUtils;
import com.digiwin.dap.middleware.dmc.internal.DMCConfig;
import com.digiwin.dap.middleware.dmc.internal.DMCConfigBuilder;
import com.digiwin.dap.middleware.dmc.internal.DMCConstants;
import com.digiwin.dap.middleware.dmc.internal.DMCHeaders;
import com.digiwin.dap.middleware.dmc.model.IamUserToken;
import com.digiwin.dmc.sdk.util.StringUtil;
import org.apache.hc.client5.http.classic.methods.HttpUriRequestBase;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.client5.http.io.HttpClientConnectionManager;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.Header;
import org.apache.hc.core5.http.io.SocketConfig;
import org.apache.hc.core5.io.CloseMode;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * Default implementation of {@link ServiceClient}.
 *
 * @author fobgochod
 * @since 3.1.0.0
 */
public class DefaultServiceClient extends ServiceClient {

    protected static HttpRequestFactory httpRequestFactory = new HttpRequestFactory();
    protected CloseableHttpClient httpClient;
    protected HttpClientConnectionManager connectionManager;
    protected RequestConfig requestConfig;
    protected DMCConfig config;
    protected AuthCache authCache = new AuthCache();
    protected AppCache appCache = new AppCache();

    public DefaultServiceClient() {
        this(DMCConfigBuilder.create().build());
    }

    public DefaultServiceClient(DMCConfig config) {
        this(config, new ClientConfiguration());
    }

    public DefaultServiceClient(DMCConfig config, ClientConfiguration clientConfig) {
        super(clientConfig);
        this.connectionManager = createHttpClientConnectionManager();
        this.httpClient = createHttpClient(this.connectionManager);

        RequestConfig.Builder builder = RequestConfig.custom();
        builder.setConnectTimeout(clientConfig.getConnectionTimeout(), TimeUnit.MILLISECONDS);
        builder.setResponseTimeout(clientConfig.getSocketTimeout(), TimeUnit.MILLISECONDS);
        builder.setConnectionRequestTimeout(clientConfig.getConnectionRequestTimeout(), TimeUnit.MILLISECONDS);
        this.requestConfig = builder.build();
        this.config = config;
    }


    private static ResponseMessage buildResponse(RequestMessage request, CloseableHttpResponse httpResponse)
            throws IOException {

        assert (httpResponse != null);

        ResponseMessage response = new ResponseMessage();
        response.setUrl(request.getEndpoint().toString());

        if (httpResponse.getCode() != 0) {
            response.setStatusCode(httpResponse.getCode());
        }

        if (httpResponse.getEntity() != null) {
            response.setContent(httpResponse.getEntity().getContent());
            if (!response.isSuccessful()) {
                String content = StreamUtils.copyToString(response.getContent(), StandardCharsets.UTF_8);
                response.setError(content);
            }
        }

        for (Header header : httpResponse.getHeaders()) {
            if (HttpHeaders.CONTENT_LENGTH.equalsIgnoreCase(header.getName())) {
                response.setContentLength(Long.parseLong(header.getValue()));
            }
            response.addHeader(header.getName(), header.getValue());
        }

        HttpUtil.convertHeaderCharsetFromIso88591(response.getHeaders());

        return response;
    }

    @Override
    public ResponseMessage sendRequestCore(RequestMessage request, ExecutionContext context) throws IOException {
        HttpUriRequestBase httpRequest = httpRequestFactory.createHttpRequest(request);
        bindAppIds(request);
        handleCommonToken(request, httpRequest);
        HttpClientContext httpContext = createHttpContext(request);

        CloseableHttpResponse httpResponse;
        try {
            httpResponse = httpClient.execute(httpRequest, httpContext);
        } catch (IOException ex) {
            httpRequest.abort();
            throw ExceptionFactory.createNetworkException(ex);
        }

        return buildResponse(request, httpResponse);
    }

    private void bindAppIds(RequestMessage request) {
        if (request.isWhitelist()) {
            return;
        }
        int appIdSize = appCache.getAppIdSize();
        List<String> appIds = config.getAppIds();
        if (appIdSize > 0 || appIds == null || appIds.isEmpty()) {
            return;
        }
        appCache.syncAppIds(config);
    }

    @Override
    public DMCConfig getConfig() {
        return config;
    }

    @Override
    public AuthCache getAuthCache() {
        return this.authCache;
    }

    @Override
    public synchronized void setAuthCache(String tenantId) {
        this.config.setTenantId(tenantId);
        this.authCache.getToken(config);
    }

    @Override
    public AppCache getAppCache() {
        return this.appCache;
    }

    @Override
    public void shutdown() {
        IdleConnectionReaper.remove(this.connectionManager);
        this.connectionManager.close(CloseMode.GRACEFUL);
    }

    @Override
    public String getConnectionPoolStats() {
        if (connectionManager != null && connectionManager instanceof PoolingHttpClientConnectionManager) {
            PoolingHttpClientConnectionManager conn = (PoolingHttpClientConnectionManager) connectionManager;
            return conn.getTotalStats().toString();
        }
        return "";
    }

    private HttpClientConnectionManager createHttpClientConnectionManager() {
        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
        connectionManager.setDefaultMaxPerRoute(clientConfig.getMaxConnections());
        connectionManager.setMaxTotal(clientConfig.getMaxConnections());
        connectionManager.setDefaultSocketConfig(SocketConfig.custom().setSoTimeout(clientConfig.getSocketTimeout(), TimeUnit.MILLISECONDS).build());
        if (clientConfig.isUseReaper()) {
            IdleConnectionReaper.setIdleConnectionTime(clientConfig.getIdleConnectionTime());
            IdleConnectionReaper.register(connectionManager);
        }
        return connectionManager;
    }

    private CloseableHttpClient createHttpClient(HttpClientConnectionManager connectionManager) {
        return HttpClients.custom().setConnectionManager(connectionManager)
                .disableContentCompression()
                .setRetryStrategy(clientConfig.getRetryHandler())
                .addResponseInterceptorLast(clientConfig.getResponseInterceptor())
                .build();
    }

    private void handleCommonToken(RequestMessage request, HttpUriRequestBase httpRequest) {
        if (!request.isWhitelist()) {
            IamUserToken iamUserToken = IamUserTokenHolder.getContext();
            if (!request.isOnlySysAccountAccess()
                    && Objects.nonNull(iamUserToken)
                    && !StringUtil.isEmpty(iamUserToken.getUserToken())) {
                httpRequest.setHeader(DMCHeaders.HTTP_HEADER_USER_TOKEN_KEY, iamUserToken.getUserToken());
            } else {
                httpRequest.setHeader(DMCHeaders.HTTP_HEADER_USER_TOKEN_KEY, authCache.getToken(config));
            }
            if (Objects.nonNull(iamUserToken) && iamUserToken.isUseOnce()) {
                IamUserTokenHolder.clearContext();
            }

            if (DMCConstants.DEFAULT_TENANT.equals(config.getTenantId()) && TenantHolder.getContext() != null) {
                // 默认租户才允许任意设定文件租户信息，其他租户默认取当前登录租户信息
                putHeaderIfAbsent(httpRequest, DMCHeaders.HTTP_HEADER_ACCESS_TOKEN_KEY, DMCConstants.DMC_SDK + TenantHolder.getContext());
            }
        }
        putHeaderIfAbsent(httpRequest, DMCHeaders.HTTP_HEADER_APP_TOKEN_KEY, config.getAppToken());
        putHeaderIfAbsent(httpRequest, DMCHeaders.HTTP_HEADER_APP_SECRET, config.getAppSecret());
    }

    private void putHeaderIfAbsent(HttpUriRequestBase httpRequest, String name, String value) {
        if (!httpRequest.containsHeader(name)) {
            httpRequest.setHeader(name, value);
        }
    }

    private HttpClientContext createHttpContext(RequestMessage request) {
        HttpClientContext httpContext = HttpClientContext.create();
        httpContext.setRequestConfig(this.requestConfig);
        httpContext.setAttribute(CustomHttpRequestRetryHandler.RETRY_ATTRIBUTE, request.isForceRetry());
        return httpContext;
    }
}
