package com.digiwin.dap.middleware.dmc.common.comm;

import org.apache.hc.client5.http.io.HttpClientConnectionManager;
import org.apache.hc.core5.pool.ConnPoolControl;
import org.apache.hc.core5.util.TimeValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * A daemon thread used to periodically check connection pools for idle
 * * connections.
 *
 * @author fobgochod
 * @since 3.1.0.0
 */
public final class IdleConnectionReaper extends Thread {

    private static final Logger logger = LoggerFactory.getLogger(IdleConnectionReaper.class);
    private static final int REAP_INTERVAL_MILLISECONDS = 5 * 1000;
    private static final List<HttpClientConnectionManager> connectionManagers = new ArrayList<>();

    private static IdleConnectionReaper INSTANCE;

    private static long idleConnectionTime = 60 * 1000;

    private volatile boolean shuttingDown;

    private IdleConnectionReaper() {
        super("idle_connection_reaper");
        setDaemon(true);
    }

    public static synchronized void register(HttpClientConnectionManager connectionManager) {
        if (INSTANCE == null) {
            INSTANCE = new IdleConnectionReaper();
            INSTANCE.start();
        }
        connectionManagers.add(connectionManager);
    }

    public static synchronized void remove(HttpClientConnectionManager connectionManager) {
        connectionManagers.remove(connectionManager);
        if (connectionManagers.isEmpty()) {
            shutdown();
        }
    }

    public static synchronized void shutdown() {
        if (INSTANCE != null) {
            INSTANCE.markShuttingDown();
            INSTANCE.interrupt();
            INSTANCE = null;
        }
        connectionManagers.clear();
    }

    public static synchronized int size() {
        return connectionManagers.size();
    }

    public static synchronized void setIdleConnectionTime(long idleTime) {
        idleConnectionTime = idleTime;
    }

    private void markShuttingDown() {
        shuttingDown = true;
    }

    @Override
    public void run() {
        while (true) {
            if (shuttingDown) {
                logger.debug("Shutting down reaper thread.");
                return;
            }

            try {
                Thread.sleep(REAP_INTERVAL_MILLISECONDS);
            } catch (InterruptedException e) {
            }

            try {
                List<HttpClientConnectionManager> connectionManagers;
                synchronized (IdleConnectionReaper.class) {
                    connectionManagers = new ArrayList<>(IdleConnectionReaper.connectionManagers);
                }
                for (HttpClientConnectionManager connectionManager : connectionManagers) {
                    try {
                        if (connectionManager instanceof ConnPoolControl) {
                            ConnPoolControl<?> pool = (ConnPoolControl<?>) connectionManager;
                            pool.closeExpired();
                            pool.closeIdle(TimeValue.ofMilliseconds(idleConnectionTime));
                        }
                    } catch (Exception ex) {
                        logger.warn("Unable to close idle connections", ex);
                    }
                }
            } catch (Throwable t) {
                logger.debug("Reaper thread: ", t);
            }
        }
    }
}
