package com.digiwin.dap.middleware.dmc.common.utils;

import com.digiwin.dap.middleware.dmc.ClientErrorCode;
import com.digiwin.dap.middleware.dmc.ClientException;
import com.digiwin.dap.middleware.dmc.DMCErrorCode;
import com.digiwin.dap.middleware.dmc.DMCException;
import com.digiwin.dap.middleware.dmc.internal.DMCUtils;
import com.digiwin.dap.middleware.dmc.internal.model.StdError;
import org.apache.hc.client5.http.ClientProtocolException;
import org.apache.hc.client5.http.ConnectTimeoutException;
import org.apache.hc.client5.http.HttpHostConnectException;
import org.apache.hc.core5.http.NoHttpResponseException;

import java.io.IOException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.nio.channels.NonReadableChannelException;

/**
 * A simple factory used for creating instances of <code>ClientException</code>
 * and <code>DMCException</code>.
 *
 * @author fobgochod
 * @since 3.1.0.0
 */
public class ExceptionFactory {

    public static ClientException createNetworkException(IOException ex) {
        String requestId = "Unknown";
        String errorCode = ClientErrorCode.UNKNOWN;

        if (ex instanceof SocketTimeoutException) {
            errorCode = ClientErrorCode.SOCKET_TIMEOUT;
        } else if (ex instanceof SocketException) {
            errorCode = ClientErrorCode.SOCKET_EXCEPTION;
        } else if (ex instanceof ConnectTimeoutException) {
            errorCode = ClientErrorCode.CONNECTION_TIMEOUT;
        } else if (ex instanceof UnknownHostException) {
            errorCode = ClientErrorCode.UNKNOWN_HOST;
        } else if (ex instanceof HttpHostConnectException) {
            errorCode = ClientErrorCode.CONNECTION_REFUSED;
        } else if (ex instanceof NoHttpResponseException) {
            errorCode = ClientErrorCode.CONNECTION_TIMEOUT;
        } else if (ex instanceof ClientProtocolException) {
            Throwable cause = ex.getCause();
            if (cause instanceof NonReadableChannelException) {
                errorCode = ClientErrorCode.NONREPEATABLE_REQUEST;
                return new ClientException(cause.getMessage(), errorCode, requestId, cause);
            }
        }
        return new ClientException(ex.getMessage(), errorCode, requestId, ex);
    }

    public static DMCException createInvalidResponseException(String requestId, String rawResponseError,
                                                              Throwable cause) {
        return createInvalidResponseException(requestId,
                DMCUtils.COMMON_RESOURCE_MANAGER.getFormattedString("FailedToParseResponse", cause.getMessage()),
                rawResponseError);
    }

    public static DMCException createInvalidResponseException(String requestId, String message,
                                                              String rawResponseError) {
        return createDMCException(requestId, DMCErrorCode.INVALID_RESPONSE, message, rawResponseError);
    }

    public static DMCException createDMCException(String requestId, String errorCode, String message,
                                                  String rawResponseError) {
        return new DMCException(message, errorCode, requestId, null, null, null, null, rawResponseError);
    }

    public static DMCException createDMCException(String requestId, String errorCode, String message) {
        return new DMCException(message, errorCode, requestId, null, null, null, null);
    }

    public static DMCException createUnknownDMCException(String requestId, int statusCode) {
        String message = "No body in response, http status code " + statusCode;
        return new DMCException(message, ClientErrorCode.UNKNOWN, requestId, null, null, null, null);
    }

    public static DMCException createDMCException(StdError errorResult) {
        return new DMCException(String.format("%s：%s", errorResult.getErrorMessage(), errorResult.getMessage()),
                errorResult.getErrorCode());
    }
}
