package com.digiwin.dap.middleware.dmc.internal;

import com.digiwin.dap.middleware.dmc.ClientException;
import com.digiwin.dap.middleware.dmc.DMCErrorCode;
import com.digiwin.dap.middleware.dmc.DMCException;
import com.digiwin.dap.middleware.dmc.common.comm.ResponseHandler;
import com.digiwin.dap.middleware.dmc.common.comm.ResponseMessage;
import com.digiwin.dap.middleware.dmc.common.parser.ErrorResponseParser;
import com.digiwin.dap.middleware.dmc.common.parser.ResponseParseException;
import com.digiwin.dap.middleware.dmc.common.utils.ExceptionFactory;
import com.digiwin.dap.middleware.dmc.internal.model.StdError;
import org.apache.hc.core5.http.HttpStatus;

/**
 * error response handler
 *
 * @author fobgochod
 * @since 3.1.0.0
 */
public class ErrorResponseHandler implements ResponseHandler {

    @Override
    public void handle(ResponseMessage response) throws DMCException, ClientException {

        if (response.isSuccessful()) {
            return;
        }

        String requestId = response.getRequestId();
        int statusCode = response.getStatusCode();
        if (response.getContent() == null) {
            /**
             * When HTTP response body is null, handle status code 404 Not
             * Found, 304 Not Modified, 412 Precondition Failed especially.
             */
            if (statusCode == HttpStatus.SC_NOT_FOUND) {
                throw ExceptionFactory.createDMCException(requestId, DMCErrorCode.NO_SUCH_KEY, "Not Found");
            } else if (statusCode == HttpStatus.SC_NOT_MODIFIED) {
                throw ExceptionFactory.createDMCException(requestId, DMCErrorCode.NOT_MODIFIED, "Not Modified");
            } else if (statusCode == HttpStatus.SC_PRECONDITION_FAILED) {
                throw ExceptionFactory.createDMCException(requestId, DMCErrorCode.PRECONDITION_FAILED, "Precondition Failed");
            } else if (statusCode == HttpStatus.SC_FORBIDDEN) {
                throw ExceptionFactory.createDMCException(requestId, DMCErrorCode.ACCESS_FORBIDDEN, "AccessForbidden");
            } else {
                throw ExceptionFactory.createUnknownDMCException(requestId, statusCode);
            }
        }

        try {
            ErrorResponseParser parser = new ErrorResponseParser();
            StdError stdError = parser.parse(response);
            throw ExceptionFactory.createDMCException(stdError);
        } catch (ResponseParseException e) {
            throw ExceptionFactory.createInvalidResponseException(requestId, response.getError(), e);
        } finally {
            DMCUtils.safeCloseResponse(response);
        }
    }
}
