package com.digiwin.dap.middleware.dmc.internal.client;

import com.digiwin.dap.middleware.dmc.DMCException;
import com.digiwin.dap.middleware.dmc.common.comm.ServiceClient;
import com.digiwin.dap.middleware.dmc.common.context.IamUserTokenHolder;
import com.digiwin.dap.middleware.dmc.common.utils.StreamUtils;
import com.digiwin.dap.middleware.dmc.internal.client.model.ShareV1Request;
import com.digiwin.dap.middleware.dmc.internal.client.model.UploadPartV1Request;
import com.digiwin.dap.middleware.dmc.internal.client.model.UploadV1Request;
import com.digiwin.dap.middleware.dmc.internal.client.operation.*;
import com.digiwin.dap.middleware.dmc.internal.operation.DMCBucketOperation;
import com.digiwin.dap.middleware.dmc.model.*;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.List;


public class OSSClient implements OSS {

    protected ServiceClient serviceClient;

    private UploadV1Operation uploadV1Operation;
    private DownloadV1Operation downloadV1Operation;
    private DeleteV1Operation deleteV1Operation;
    private ShareV1Operation shareV1Operation;
    private DirInfoV1Operation dirInfoV1Operation;
    private FileInfoV1Operation fileInfoV1Operation;
    private DMCBucketOperation dmcBucketOperation;

    protected void initOperations() {
        this.uploadV1Operation = new UploadV1Operation(serviceClient);
        this.downloadV1Operation = new DownloadV1Operation(serviceClient);
        this.deleteV1Operation = new DeleteV1Operation(serviceClient);
        this.shareV1Operation = new ShareV1Operation(serviceClient);
        this.dirInfoV1Operation = new DirInfoV1Operation(serviceClient);
        this.fileInfoV1Operation = new FileInfoV1Operation(serviceClient);
        this.dmcBucketOperation = new DMCBucketOperation(serviceClient);
    }

    @Override
    public FileInfo uploadV1(InputStream input, FileInfo fileInfo) {
        UploadV1Request request = new UploadV1Request();
        request.setInput(input);
        request.setFileInfo(fileInfo);
        return uploadV1(request);
    }

    @Override
    public FileInfo uploadV1(UploadV1Request request) {
        try {
            IamUserToken iamUserToken = IamUserTokenHolder.getContext();
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            if (iamUserToken != null && iamUserToken.isUseOnce()) {
                IamUserTokenHolder.setContext(iamUserToken);
            }
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1Operation.upload(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public FileInfo uploadCoverV1(String fileId, InputStream input) {
        UploadV1Request request = new UploadV1Request();
        request.setInput(input);
        request.setFileId(fileId);
        return uploadCoverV1(request);
    }

    @Override
    public FileInfo uploadCoverV1(UploadV1Request request) {
        try {
            IamUserToken iamUserToken = IamUserTokenHolder.getContext();
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            if (iamUserToken != null && iamUserToken.isUseOnce()) {
                IamUserTokenHolder.setContext(iamUserToken);
            }
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1Operation.uploadCover(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public ShareInfo uploadShareV1(InputStream input, FileInfo fileInfo) {
        UploadV1Request request = new UploadV1Request();
        request.setInput(input);
        request.setFileInfo(fileInfo);
        return uploadShareV1(request);
    }

    @Override
    public ShareInfo uploadShareV1(UploadV1Request request) {
        try {
            IamUserToken iamUserToken = IamUserTokenHolder.getContext();
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            if (iamUserToken != null && iamUserToken.isUseOnce()) {
                IamUserTokenHolder.setContext(iamUserToken);
            }
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1Operation.uploadShare(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String claimFileIdV1(UploadPartV1Request request) {
        try {
            IamUserToken iamUserToken = IamUserTokenHolder.getContext();
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            if (iamUserToken != null && iamUserToken.isUseOnce()) {
                IamUserTokenHolder.setContext(iamUserToken);
            }
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1Operation.claimFileId(request).getId();
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String uploadPartV1(UploadPartV1Request request) {
        try {
            IamUserToken iamUserToken = IamUserTokenHolder.getContext();
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            if (iamUserToken != null && iamUserToken.isUseOnce()) {
                IamUserTokenHolder.setContext(iamUserToken);
            }
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1Operation.uploadPart(request).getId();
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String uploadPartCoverV1(UploadPartV1Request request) {
        try {
            IamUserToken iamUserToken = IamUserTokenHolder.getContext();
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            if (iamUserToken != null && iamUserToken.isUseOnce()) {
                IamUserTokenHolder.setContext(iamUserToken);
            }
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1Operation.uploadPartCover(request).getId();
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void downloadV1(String fileId, OutputStream output) {
        DownloadRequest request = new DownloadRequest();
        request.setFileId(fileId);
        request.setOutput(output);
        downloadV1(request);
    }

    @Override
    public byte[] downloadV1(DownloadRequest request) {
        try {
            byte[] bytes = downloadV1Operation.download(request);
            if (request.getOutput() != null) {
                StreamUtils.copy(bytes, request.getOutput());
            }
            return bytes;
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public byte[] downloadPartV1(DownloadPartRequest request) {
        try {
            byte[] bytes = downloadV1Operation.downloadPart(request);
            StreamUtils.copy(bytes, request.getOutput());
            return bytes;
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String removeV1(String fileId) {
        BatchRequest request = new BatchRequest();
        request.setFileId(fileId);
        return removeV1(request);
    }

    @Override
    public String removeV1(BatchRequest request) {
        try {
            return deleteV1Operation.delete(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String removeDirV1(String dirId) {
        BatchRequest request = new BatchRequest();
        request.setDirId(dirId);
        return removeDirV1(request);
    }

    @Override
    public String removeDirV1(BatchRequest request) {
        try {
            return deleteV1Operation.deleteDir(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void batchRemoveV1(List<String> fileIds, List<String> dirIds) {
        BatchRequest request = new BatchRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        batchRemoveV1(request);
    }

    @Override
    public void batchRemoveV1(BatchRequest request) {
        try {
            deleteV1Operation.deleteMulti(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void deleteV1(String fileId) {
        BatchRequest request = new BatchRequest();
        request.setFileId(fileId);
        deleteV1(request);
    }

    @Override
    public void deleteV1(BatchRequest request) {
        try {
            deleteV1Operation.deleteForce(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String shareV1(String fileId) {
        List<String> shareUrls = shareV1(Collections.singletonList(fileId));
        return shareUrls.isEmpty() ? null : shareUrls.get(0);
    }

    @Override
    public List<String> shareV1(List<String> fileIds) {
        ShareV1Request request = new ShareV1Request();
        request.setFileIds(fileIds);
        return shareV1(request);
    }

    @Override
    public List<String> shareV1(List<String> fileIds, long expireDay) {
        ShareV1Request request = new ShareV1Request();
        request.setFileIds(fileIds);
        request.setExpireDay(expireDay);
        return shareV1(request);
    }

    @Override
    public List<String> shareV1(ShareV1Request request) {
        try {
            if (request.getExpireDay() > 0) {
                return shareV1Operation.shareDay(request);
            } else {
                return shareV1Operation.share(request);
            }
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void deleteShareV1(String shareId) {
        ShareV1Request request = new ShareV1Request();
        request.setShareId(shareId);
        deleteShareV1(request);
    }

    @Override
    public void deleteShareV1(ShareV1Request request) {
        try {
            shareV1Operation.deleteShare(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }


    @Override
    public String copyV1(String fileId, String targetDirId) {
        FileOpRequest request = new FileOpRequest();
        request.setFileId(fileId);
        request.setTargetDirId(targetDirId);
        return copyV1(request);
    }

    @Override
    public String copyV1(FileOpRequest request) {
        try {
            return fileInfoV1Operation.copyFile(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String copyDirV1(String dirId, String targetDirId) {
        FileOpRequest request = new FileOpRequest();
        request.setDirId(dirId);
        request.setTargetDirId(targetDirId);
        return copyDirV1(request);
    }

    @Override
    public String copyDirV1(FileOpRequest request) {
        try {
            return dirInfoV1Operation.copyDir(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void batchCopyV1(List<String> fileIds, List<String> dirIds, String targetDirId) {
        FileOpRequest request = new FileOpRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        request.setTargetDirId(targetDirId);
        batchCopyV1(request);
    }

    @Override
    public void batchCopyV1(FileOpRequest request) {
        try {
            fileInfoV1Operation.copyFileBatch(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void moveV1(String fileId, String targetDirId) {
        FileOpRequest request = new FileOpRequest();
        request.setFileId(fileId);
        request.setTargetDirId(targetDirId);
        moveV1(request);
    }

    @Override
    public void moveV1(FileOpRequest request) {
        try {
            fileInfoV1Operation.moveFile(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void moveDirV1(String dirId, String targetDirId) {
        FileOpRequest request = new FileOpRequest();
        request.setDirId(dirId);
        request.setTargetDirId(targetDirId);
        moveDirV1(request);
    }

    @Override
    public void moveDirV1(FileOpRequest request) {
        try {
            dirInfoV1Operation.moveDir(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void batchMoveV1(List<String> fileIds, List<String> dirIds, String targetDirId) {
        FileOpRequest request = new FileOpRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        request.setTargetDirId(targetDirId);
        batchMoveV1(request);
    }

    @Override
    public void batchMoveV1(FileOpRequest request) {
        try {
            fileInfoV1Operation.moveFileBatch(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String createDirV1(String dirName) {
        DirRequest request = new DirRequest();
        request.setDirInfo(new DirInfo(dirName));
        return createDirV1(request);
    }

    @Override
    public String createDirV1(String dirName, String parentId) {
        DirRequest request = new DirRequest();
        request.setDirInfo(new DirInfo(parentId, dirName));
        return createDirV1(request);
    }

    @Override
    public String createDirV1(DirRequest request) {
        try {
            return dirInfoV1Operation.createDir(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void renameDirV1(String dirId, String dirName) {
        DirRequest request = new DirRequest();
        request.setDirId(dirId);
        request.setDirName(dirName);
        renameDirV1(request);
    }

    @Override
    public void renameDirV1(DirRequest request) {
        try {
            dirInfoV1Operation.renameDir(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public DirInfo getDirInfoV1(String dirId) {
        DirRequest request = new DirRequest();
        request.setDirId(dirId);
        return getDirInfoV1(request);
    }

    @Override
    public DirInfo getDirInfoV1(DirRequest request) {
        try {
            return dirInfoV1Operation.getDirInfo(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public DirInfo getDirInfoByNameV1(String parentDirId, String dirName) {
        DirRequest request = new DirRequest();
        DirInfo dirInfo = new DirInfo();
        dirInfo.setParentId(parentDirId);
        dirInfo.setName(dirName);
        request.setDirInfo(dirInfo);
        try {
            return dirInfoV1Operation.getDirInfoByName(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public FolderInfo getDirContentV1(String dirId) {
        DirRequest request = new DirRequest();
        request.setDirId(dirId);
        return getDirContentV1(request);
    }

    @Override
    public FolderInfo getDirContentV1(DirRequest request) {
        try {
            return dirInfoV1Operation.getDirContent(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void renameV1(String fileId, String fileName) {
        FileInfoRequest request = new FileInfoRequest();
        request.setFileId(fileId);
        request.setFileName(fileName);
        renameV1(request);
    }

    @Override
    public void renameV1(FileInfoRequest request) {
        try {
            fileInfoV1Operation.renameFile(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public FileInfo getFileInfoV1(String fileId) {
        FileInfoRequest request = new FileInfoRequest();
        request.setFileId(fileId);
        return getFileInfoV1(request);
    }

    @Override
    public FileInfo getFileInfoV1(FileInfoRequest request) {
        try {
            return fileInfoV1Operation.getFileInfo(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String getToken(String username, String pwdhash, String tenantId) {
        return serviceClient.getAuthCache().getToken(username, pwdhash, tenantId);
    }
}
