package com.digiwin.dap.middleware.dmc.internal.operation;

import com.digiwin.dap.middleware.dmc.DMCException;
import com.digiwin.dap.middleware.dmc.HttpMethod;
import com.digiwin.dap.middleware.dmc.common.comm.RequestMessage;
import com.digiwin.dap.middleware.dmc.common.comm.ServiceClient;
import com.digiwin.dap.middleware.dmc.common.utils.StreamUtils;
import com.digiwin.dap.middleware.dmc.event.ProgressEventType;
import com.digiwin.dap.middleware.dmc.event.ProgressListener;
import com.digiwin.dap.middleware.dmc.event.ProgressPublisher;
import com.digiwin.dap.middleware.dmc.internal.DMCConstants;
import com.digiwin.dap.middleware.dmc.internal.DMCOperation;
import com.digiwin.dap.middleware.dmc.internal.DMCRequestMessageBuilder;
import com.digiwin.dap.middleware.dmc.internal.ResponseParsers;
import com.digiwin.dap.middleware.dmc.model.*;

import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.List;

public class DMCUploadOperation extends DMCOperation {

    public DMCUploadOperation(ServiceClient client) {
        super(client);
    }

    public FileInfo upload(UploadRequest original) throws Exception {
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getUploadUrl())
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setEntity(original.getEntity())
                .setInputSize(original.getEntity().getContentLength())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .build();
        return doOperation(request, ResponseParsers.uploadResponseParser, true);
    }

    public FileInfo coverUpload(UploadCoverRequest original) throws Exception {
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getUploadCoverUrl())
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setEntity(original.getEntity())
                .setInputSize(original.getEntity().getContentLength())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .build();
        return doOperation(request, ResponseParsers.uploadResponseParser, true);
    }

    public List<FileInfo> uploadMulti(UploadMultiRequest original) throws Exception {
        RequestMessage requestMessage = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getUploadMultiUrl())
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setEntity(original.getEntity())
                .setInputSize(original.getEntity().getContentLength())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .build();
        return doOperation(requestMessage, ResponseParsers.uploadMultiResponseParser);
    }

    public String multipartUpload(MultipartUploadRequest request) throws Exception {
        final long partSize = request.getPartSize();
        if (partSize % DMCConstants.DEFAULT_CHUNK_SIZE != 0) {
            throw new DMCException("分段大小请设置261120的整数倍");
        }
        /*
         * Claim a file id firstly
         */
        String fileId = claimFileId(request.getFileInfo());
        logger.info("Claiming a new file id {}.", fileId);

        long fileLength = request.getFileLength();
        int partCount = (int) (fileLength / partSize);
        if (fileLength % partSize != 0) {
            partCount++;
        }
        if (partCount > 10000) {
            throw new DMCException("分段总数不能超过10000");
        } else {
            logger.info("Total parts count {}.", partCount);
        }

        final ProgressListener listener = request.getProgressListener();
        ProgressPublisher.publishProgress(listener, ProgressEventType.TRANSFER_STARTED_EVENT);
        ProgressPublisher.publishRequestContentLength(listener, fileLength);

        if (request.getInput() != null) {
            try (InputStream inputStream = request.getInput()) {
                for (int i = 0; i < partCount; i++) {
                    long startPos = i * partSize;
                    int curPartSize = (int) ((i + 1 == partCount) ? (fileLength - startPos) : partSize);

                    byte[] buffer = new byte[curPartSize];
                    inputStream.read(buffer);

                    uploadPart(fileId, buffer, startPos, curPartSize, fileLength, partCount, i + 1, listener);
                }
            } catch (Exception e) {
                ProgressPublisher.publishProgress(listener, ProgressEventType.TRANSFER_FAILED_EVENT);
                throw e;
            }
        } else {
            try (RandomAccessFile raf = new RandomAccessFile(request.getFile(), "r")) {
                MappedByteBuffer map = raf.getChannel().map(FileChannel.MapMode.READ_ONLY, 0, fileLength);
                for (int i = 0; i < partCount; i++) {
                    long startPos = i * partSize;
                    int curPartSize = (int) ((i + 1 == partCount) ? (fileLength - startPos) : partSize);

                    byte[] buffer = new byte[curPartSize];
                    map.get(buffer, 0, curPartSize);

                    uploadPart(fileId, buffer, startPos, curPartSize, fileLength, partCount, i + 1, listener);
                }
            } catch (Exception e) {
                ProgressPublisher.publishProgress(listener, ProgressEventType.TRANSFER_FAILED_EVENT);
                throw e;
            }
        }
        ProgressPublisher.publishProgress(listener, ProgressEventType.TRANSFER_COMPLETED_EVENT);
        return fileId;
    }

    private void uploadPart(String fileId, byte[] buffer, long startPos, int curPartSize, long fileLength,
                            int partCount, int partNum, ProgressListener listener) throws Exception {
        PartUploader partUploader = new PartUploader();
        partUploader.setFileId(fileId);
        partUploader.setParts(buffer);
        partUploader.setFrom(startPos);
        partUploader.setTo(startPos + curPartSize - 1);
        partUploader.setTotal(fileLength);
        partUploader.setPartCount(partCount);
        partUploader.setPartNum(partNum);
        uploadPart(partUploader);

        ProgressPublisher.publishRequestBytesTransferred(listener, curPartSize);
        logger.info("总共：{}段, 第{}段上传成功：fileId={},startPos={},curPartSize={}.", partCount, partNum, fileId, startPos, curPartSize);
    }

    private void uploadPart(PartUploader original) throws Exception {
        RequestMessage requestMessage = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getUploadMultipartUrl(original.getFileId(), original.getFrom(), original.getTo(), original.getTotal(), original.getPartNum()))
                .setMethod(HttpMethod.POST)
                .setEntity(original.getEntity())
                .setForceRetry(true)
                .build();
        doOperation(requestMessage, ResponseParsers.uploadMultipartResponseParser);
    }

    private String claimFileId(FileInfo fileInfo) throws Exception {
        FileInfoRequest original = new FileInfoRequest();
        original.setFileInfo(fileInfo);

        RequestMessage requestMessage = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getUploadEmptyUrl())
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setEntity(original.getEntity())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .build();
        return doOperation(requestMessage, ResponseParsers.uploadMultipartResponseParser);
    }
}


