package com.digiwin.dmc.sdk.service;

import com.digiwin.dap.middleware.dmc.DMC;
import com.digiwin.dap.middleware.dmc.HttpMethod;
import com.digiwin.dap.middleware.dmc.common.context.TenantId;
import com.digiwin.dmc.sdk.entity.BatchOperationIds;
import com.digiwin.dmc.sdk.entity.ContentsResult;
import com.digiwin.dmc.sdk.entity.DirInfo;
import com.digiwin.dmc.sdk.entity.FileInfo;
import com.digiwin.dmc.sdk.service.discard.IFileInfoTextService;
import com.digiwin.dmc.sdk.service.download.IGeneralDocumentDownloader;
import com.digiwin.dmc.sdk.service.download.ISegmentDocumentDownloader;
import com.digiwin.dmc.sdk.service.upload.ICoverGeneralDocumentUploader;
import com.digiwin.dmc.sdk.service.upload.ICoverSegmentDocumentUploader;
import com.digiwin.dmc.sdk.service.upload.IGeneralDocumentUploader;
import com.digiwin.dmc.sdk.service.upload.ISegmentDocumentUploader;

/**
 * 文件操作
 *
 * @author chenzwd
 * @date 2018-06-14 13:59
 */
public interface IDocumentStorageService extends IFileInfoTextService {


    /*****************************************************************************************************************
     *                                                   文件上传                                                      *
     *****************************************************************************************************************/

    /**
     * 获取普通文档上传器
     * 通过 {@link FileInfo#setTenantId(String)} 指定文件上传tenantId
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files
     *
     * @param bytes    文件字节流
     * @param fileInfo 文件信息
     * @return 文档上上传器
     */
    IGeneralDocumentUploader uploadDocument(byte[] bytes, FileInfo fileInfo);

    IGeneralDocumentUploader uploadDocument(String bucketName, byte[] bytes, FileInfo fileInfo);

    IGeneralDocumentUploader uploadDocument(String bucketName, byte[] bytes, FileInfo fileInfo, String driveToken);

    IGeneralDocumentUploader uploadDocument(String localPath, FileInfo fileInfo);

    IGeneralDocumentUploader uploadDocument(String bucketName, String localPath, FileInfo fileInfo);

    IGeneralDocumentUploader uploadDocument(String bucketName, String localPath, FileInfo fileInfo, String driveToken);


    /**
     * 获取分段文档上传器
     * 通过 {@link FileInfo#setTenantId(String)} 指定文件上传tenantId
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files/segment
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files/{fileId}/{from}/{to}/{total}
     *
     * @param bytes    文件字节流
     * @param fileInfo 文件信息
     * @return 文档上上传器
     */
    ISegmentDocumentUploader getUploader(byte[] bytes, FileInfo fileInfo);

    ISegmentDocumentUploader getUploader(String bucketName, byte[] bytes, FileInfo fileInfo);

    ISegmentDocumentUploader getUploader(String bucketName, byte[] bytes, FileInfo fileInfo, String driveToken);

    ISegmentDocumentUploader getUploader(String localPath, FileInfo fileInfo);

    ISegmentDocumentUploader getUploader(String bucketName, String localPath, FileInfo fileInfo);

    ISegmentDocumentUploader getUploader(String bucketName, String localPath, FileInfo fileInfo, String driveToken);


    /**
     * 获取普通文档上传器(覆盖)
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files/{fileId}/cover
     *
     * @param bytes    文件字节流
     * @param fileId   文件id
     * @param tenantId 租户ID
     * @return 文档上传器
     */
    ICoverGeneralDocumentUploader coverUploadDocument(byte[] bytes, String fileId, TenantId... tenantId);

    ICoverGeneralDocumentUploader coverUploadDocument(String bucketName, byte[] bytes, String fileId, TenantId... tenantId);

    ICoverGeneralDocumentUploader coverUploadDocument(String bucketName, byte[] bytes, String fileId, String driveToken, TenantId... tenantId);

    ICoverGeneralDocumentUploader coverUploadDocument(String localPath, String fileId, TenantId... tenantId);

    ICoverGeneralDocumentUploader coverUploadDocument(String bucketName, String localPath, String fileId, TenantId... tenantId);

    ICoverGeneralDocumentUploader coverUploadDocument(String bucketName, String localPath, String fileId, String driveToken, TenantId... tenantId);


    /**
     * 获取分段文档上传器(覆盖)
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files/{fileId}/{from}/{to}/{total}/cover
     *
     * @param bytes  文件字节流
     * @param fileId 文件id
     * @return 文档上传器
     */
    ICoverSegmentDocumentUploader getUploader(byte[] bytes, String fileId, TenantId... tenantId);

    ICoverSegmentDocumentUploader getUploader(String bucketName, byte[] bytes, String fileId, TenantId... tenantId);

    ICoverSegmentDocumentUploader getUploader(String bucketName, byte[] bytes, String fileId, String driveToken, TenantId... tenantId);

    ICoverSegmentDocumentUploader getUploader(String localPath, String fileId, TenantId... tenantId);

    ICoverSegmentDocumentUploader getUploader(String bucketName, String localPath, String fileId, TenantId... tenantId);

    ICoverSegmentDocumentUploader getUploader(String bucketName, String localPath, String fileId, String driveToken, TenantId... tenantId);


    /*****************************************************************************************************************
     *                                                   文件下载                                                      *
     *****************************************************************************************************************/

    /**
     * 获取普通文档下载器
     * <p>
     * {@link HttpMethod#GET} - /api/dmc/v1/buckets/{bucket}/files/{fileId}
     *
     * @param fileId     文件ID
     * @param saveToPath 保存路径
     * @return 文档下载器
     */
    IGeneralDocumentDownloader downloadDocument(String fileId, String saveToPath, TenantId... tenantId);

    IGeneralDocumentDownloader downloadDocument(String bucketName, String fileId, String saveToPath, TenantId... tenantId);

    IGeneralDocumentDownloader downloadDocument(String bucketName, String fileId, String saveToPath, String driveToken, TenantId... tenantId);


    /**
     * 获取分段文档下载器
     * <p>
     * {@link HttpMethod#GET} - /api/dmc/v1/buckets/{bucket}/files/{fileId}/{position}/{length}
     *
     * @param fileId     文件ID
     * @param saveToPath 保存路径
     * @return 文档下载器
     */
    ISegmentDocumentDownloader getDownloader(String fileId, String saveToPath, TenantId... tenantId);

    ISegmentDocumentDownloader getDownloader(String bucketName, String fileId, String saveToPath, TenantId... tenantId);

    ISegmentDocumentDownloader getDownloader(String bucketName, String fileId, String saveToPath, String driveToken, TenantId... tenantId);


    /*****************************************************************************************************************
     *                                                 文件信息操作                                                    *
     *****************************************************************************************************************/

    /**
     * 删除文件(回收站)
     * <p>
     * {@link HttpMethod#DELETE} - /api/dmc/v1/buckets/{bucket}/files/{fileId}
     *
     * @param fileId 文件ID
     * @return 回收站ID
     */
    String deleteDocument(String fileId, TenantId... tenantId);

    String deleteDocument(String bucketName, String fileId, TenantId... tenantId);

    String deleteDocument(String bucketName, String fileId, String driveToken, TenantId... tenantId);

    /**
     * 删除文件
     * <p>
     * {@link HttpMethod#DELETE} - /api/dmc/v1/buckets/{bucket}/files/{fileId}/force
     *
     * @param fileId 文件ID
     */
    void deleteForce(String fileId, TenantId... tenantId);

    void deleteForce(String bucketName, String fileId, TenantId... tenantId);

    void deleteForce(String bucketName, String fileId, String driveToken, TenantId... tenantId);

    /**
     * 批量删除文件(回收站)
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files/delete/batch
     *
     * @param operationIds 文件ID集合
     */
    void deleteManyDocument(BatchOperationIds operationIds, TenantId... tenantId);

    void deleteManyDocument(String bucketName, BatchOperationIds operationIds, TenantId... tenantId);

    void deleteManyDocument(String bucketName, BatchOperationIds operationIds, String driveToken, TenantId... tenantId);


    /**
     * 重命名文件
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files/{fileId}/rename/{newFileName}
     *
     * @param fileId   文件ID
     * @param fileName 文件名称
     */
    void updateDocument(String fileId, String fileName, TenantId... tenantId);

    void updateDocument(String bucketName, String fileId, String fileName, TenantId... tenantId);

    void updateDocument(String bucketName, String fileId, String fileName, String driveToken, TenantId... tenantId);


    /**
     * 获取文件信息
     * <p>
     * {@link HttpMethod#GET} - /api/dmc/v1/buckets/{bucket}/files/{fileId}/info
     *
     * @param fileId 文件ID
     * @return 文件信息
     */
    FileInfo getDocumentInfo(String fileId, TenantId... tenantId);

    FileInfo getDocumentInfo(String bucketName, String fileId, TenantId... tenantId);

    FileInfo getDocumentInfo(String bucketName, String fileId, String driveToken, TenantId... tenantId);


    /**
     * 按照文件名称查询文件ID
     *
     * @param directoryId 目录ID，顶级目录可以传null
     * @param fileName    文件名称
     * @return 文件信息
     * @see DMC#getFileInfoByName(String, String)
     */
    String getDocumentIdByName(String directoryId, String fileName);

    /**
     * 复制文件
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files/{fileId}/copyto/directory/{targetDirId}
     *
     * @param fileId            文件ID
     * @param targetDirectoryId 目录ID
     * @return 文件ID
     */
    String copyDocument(String fileId, String targetDirectoryId, TenantId... tenantId);

    String copyDocument(String bucketName, String fileId, String targetDirectoryId, TenantId... tenantId);

    String copyDocument(String bucketName, String fileId, String targetDirectoryId, String driveToken, TenantId... tenantId);


    /**
     * 批量复制文件
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files/copy/batch/directory/{targetDirId}
     *
     * @param operationIds      文件ID集合
     * @param targetDirectoryId 目录ID
     */
    void copyManyDocument(BatchOperationIds operationIds, String targetDirectoryId, TenantId... tenantId);

    void copyManyDocument(String bucketName, BatchOperationIds operationIds, String targetDirectoryId, TenantId... tenantId);

    void copyManyDocument(String bucketName, BatchOperationIds operationIds, String targetDirectoryId, String driveToken, TenantId... tenantId);


    /**
     * 移动文件
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files/{fileId}/moveto/directory/{targetDirId}
     *
     * @param fileId            文件ID
     * @param targetDirectoryId 目录ID
     */
    void moveDocument(String fileId, String targetDirectoryId, TenantId... tenantId);

    void moveDocument(String bucketName, String fileId, String targetDirectoryId, TenantId... tenantId);

    void moveDocument(String bucketName, String fileId, String targetDirectoryId, String driveToken, TenantId... tenantId);


    /**
     * 批量移动文件
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/files/move/batch/directory/{targetDirId}
     *
     * @param operationIds      文件ID集合
     * @param targetDirectoryId 目录ID
     */
    void moveManyDocument(BatchOperationIds operationIds, String targetDirectoryId, TenantId... tenantId);

    void moveManyDocument(String bucketName, BatchOperationIds operationIds, String targetDirectoryId, TenantId... tenantId);

    void moveManyDocument(String bucketName, BatchOperationIds operationIds, String targetDirectoryId, String driveToken, TenantId... tenantId);


    /*****************************************************************************************************************
     *                                                 目录信息操作                                                    *
     *****************************************************************************************************************/

    /**
     * 创建目录
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/directorys
     *
     * @param directoryName     目录名称
     * @param parentDirectoryId 父目录ID
     * @return 目录ID
     */
    String createDirectory(String directoryName, String parentDirectoryId);

    String createDirectory(String bucketName, String directoryName, String parentDirectoryId);

    String createDirectory(String bucketName, String directoryName, String parentDirectoryId, String driveToken);


    /**
     * 删除目录
     * <p>
     * {@link HttpMethod#DELETE} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}
     *
     * @param directoryId 目录ID
     */
    String deleteDirectory(String directoryId);

    String deleteDirectory(String bucketName, String directoryId);

    String deleteDirectory(String bucketName, String directoryId, String driveToken);


    /**
     * 更新目录名称
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}/rename/{dirName}
     *
     * @param directoryId   目录ID
     * @param directoryName 目录名称
     */
    void updateDirectory(String directoryId, String directoryName);

    void updateDirectory(String bucketName, String directoryId, String directoryName);

    void updateDirectory(String bucketName, String directoryId, String directoryName, String driveToken);


    /**
     * 获取目录信息
     * <p>
     * {@link HttpMethod#GET} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}
     *
     * @param directoryId 目录ID
     * @return 目录信息
     */
    DirInfo getDirectoryInfo(String directoryId);

    DirInfo getDirectoryInfo(String bucketName, String directoryId);

    DirInfo getDirectoryInfo(String bucketName, String directoryId, String driveToken);


    /**
     * 按照目录名称、路径查询目录ID
     *
     * @param parentDirectoryId 父目录ID，顶级目录可传null
     * @param directoryName     目录名称
     * @return 目录信息
     * @see DMC#getDirInfoByName(String, String)
     */
    String getDirectoryIdByName(String parentDirectoryId, String directoryName);

    /**
     * 复制目录
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}/copyto/{targetDirId}
     *
     * @param directoryId       目录ID
     * @param targetDirectoryId 目录ID
     * @return 目录ID
     */
    String copyDirectory(String directoryId, String targetDirectoryId);

    String copyDirectory(String bucketName, String directoryId, String targetDirectoryId);

    String copyDirectory(String bucketName, String directoryId, String targetDirectoryId, String driveToken);


    /**
     * 移动目录
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}/moveto/{targetDirId}
     *
     * @param directoryId       目录ID
     * @param targetDirectoryId 目录ID
     */
    void moveDirectory(String directoryId, String targetDirectoryId);

    void moveDirectory(String bucketName, String directoryId, String targetDirectoryId);

    void moveDirectory(String bucketName, String directoryId, String targetDirectoryId, String driveToken);


    /**
     * 列出目录下的文件和子目录
     * <p>
     * {@link HttpMethod#POST} - /api/dmc/v1/buckets/{bucket}/directorys/{dirId}/list
     *
     * @param directoryId 目录ID
     * @return 文件目录信息
     */
    ContentsResult listContents(String directoryId);

    ContentsResult listContents(String bucketName, String directoryId);

    ContentsResult listContents(String bucketName, String directoryId, String driveToken);
}
