package com.digiwin.dmc.sdk.service.impl;

import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.entity.BucketInfo;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.service.IBucketManagerService;
import com.digiwin.dmc.sdk.util.HttpRequestUtil;
import com.digiwin.dmc.sdk.util.HttpUtils;
import com.digiwin.dmc.sdk.util.ObjectMapperUtil;
import com.digiwin.dmc.sdk.util.StringUtil;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class BucketManagerService implements IBucketManagerService {

    /**
     * 单一实例
     */
    public static IBucketManagerService bucketInstance() {
        return Holder.instance;
    }

    /**
     * 创建存储区
     *
     * @param bucketInfo
     * @return
     */
    @Override
    public String createBucket(BucketInfo bucketInfo) {
        return createBucketByTenant(null, bucketInfo);
    }

    /**
     * 创建存储区-多租户
     *
     * @param bucketInfo
     * @return
     */
    @Override
    public String createBucketByTenant(String discardTenantId, BucketInfo bucketInfo) {
        if (bucketInfo == null) {
            throw new IllegalArgumentException("参数bucketInfo is null");
        }
        if (StringUtil.checkSpecialId(bucketInfo.getName())) {
            throw new IllegalArgumentException("bucketName为空或包含特殊字符.号！");
        }
        Map<String, String> bucketMap = new HashMap<>();
        bucketMap.put("name", bucketInfo.getName());
        bucketMap.put("keytype", bucketInfo.getKeyType().toLowerCase());
        bucketMap.put("description", bucketInfo.getDescription());
        try {
            String uri = String.format("%s/api/dmc/v1/buckets", ServerSetting.getServiceUrl());

            Map<String, String> headers = HttpUtils.setHeader(null, discardTenantId);
            Map responseEntity = HttpRequestUtil.postJson(uri, ObjectMapperUtil.writeValueAsString(bucketMap), headers, HashMap.class);
            if (responseEntity == null || !responseEntity.containsKey("id")) {
                return null;
            } else {
                return responseEntity.get("id").toString();
            }
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    /**
     * 查询存储区
     *
     * @return
     */
    @Override
    public List<BucketInfo> getBuckets() {
        String uri = String.format("%s/api/dmc/v1/buckets", ServerSetting.getServiceUrl());
        Map<String, String> headers = HttpUtils.setHeader();
        List buckets = HttpRequestUtil.get(uri, headers, ArrayList.class);
        List<BucketInfo> bucketInfos = new ArrayList<>();
        if (buckets != null) {
            for (Object bucket : buckets) {
                try {
                    String json = ObjectMapperUtil.writeValueAsString(bucket);
                    BucketInfo bucketInfo = ObjectMapperUtil.readValue(json, BucketInfo.class);
                    bucketInfos.add(bucketInfo);
                } catch (IOException e) {
                    throw new OperateException(e);
                }
            }
        }
        return bucketInfos;
    }

    /**
     * 判断存储区是否存在
     *
     * @return
     */
    @Override
    public String isExistBucket(String bucketName) {
        return isExistBucketByTenant(null, bucketName);
    }

    /**
     * 判断存储区是否存在-多租户
     *
     * @return
     */
    @Override
    public String isExistBucketByTenant(String discardTenantId, String bucketName) {
        if (bucketName == null || ("".equals(bucketName))) {
            throw new IllegalArgumentException("参数bucketName is null");
        }
        String uri = String.format("%s/api/dmc/v1/buckets/%s/isExist", ServerSetting.getServiceUrl(), bucketName);

        Map<String, String> headers = HttpUtils.setHeaderAdmin(discardTenantId);

        Map isExist = HttpRequestUtil.get(uri, headers, HashMap.class);
        if (isExist == null || !isExist.containsKey("isExist")) {
            return null;
        }
        return isExist.get("isExist").toString();
    }

    private static class Holder { // lazy class
        static final IBucketManagerService instance = new BucketManagerService();
    }
}
