package com.digiwin.dmc.sdk.service.upload;

import com.digiwin.dap.middleware.dmc.common.context.IamUserTokenHolder;
import com.digiwin.dap.middleware.dmc.internal.client.model.UploadV1Request;
import com.digiwin.dap.middleware.dmc.model.IamUserToken;
import com.digiwin.dap.middleware.dmc.model.ShareInfo;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.entity.FileInfo;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.util.ArgumentUtils;
import com.digiwin.dmc.sdk.util.ExecutorUtil;
import com.digiwin.dmc.sdk.util.HttpUtils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Map;

/**
 * 分享并上传
 * /api/dmc/v1/buckets/{bucket}/shareUploudFile
 *
 * @date 2022/6/7 13:52
 */
public class ShareUploader implements IShareUploader {

    private String bucketName;
    private FileInfo fileInfo;
    private String driveToken;
    private InputStream inputStream;
    private IUpLoadCallbackable completeCallback;
    private UploadProgressEventArgs eventArgs = new UploadProgressEventArgs();

    private ShareUploader(String bucketName, FileInfo fileInfo, String driveToken) {
        this.bucketName = bucketName;
        this.fileInfo = fileInfo;
        this.driveToken = driveToken;
    }

    public ShareUploader(String bucketName, String localPath, FileInfo fileInfo, String driveToken) {
        this(bucketName, fileInfo, driveToken);
        try {
            this.inputStream = Files.newInputStream(Paths.get(localPath));
        } catch (IOException e) {
            throw new OperateException(e);
        }
    }

    public ShareUploader(String bucketName, byte[] bytes, FileInfo fileInfo, String driveToken) {
        this(bucketName, fileInfo, driveToken);
        this.inputStream = new ByteArrayInputStream(bytes);
    }

    /**
     * 上传Stream中的文件内容，同步模式
     *
     * @return
     */
    @Override
    public IShareUploader upload() {
        try {
            bucketName = ArgumentUtils.getBucketName(bucketName);
            Map<String, String> headers = HttpUtils.setHeader(driveToken, null, fileInfo);

            IamUserToken iamUserToken = IamUserTokenHolder.getContext();
            Runnable runnable = () -> {
                try {
                    IamUserTokenHolder.setContext(iamUserToken);
                    UploadV1Request request = new UploadV1Request();
                    request.setBucket(bucketName);
                    request.setHeaders(headers);
                    request.setInput(inputStream);
                    ShareInfo shareInfo = ServerSetting.internal().uploadShareV1(request);

                    if (completeCallback != null) {
                        eventArgs.setFileId(shareInfo.getFileId());
                        eventArgs.setUrl(shareInfo.getUrl());
                        eventArgs.setPercentage(1);
                        completeCallback.callback(eventArgs);
                    }
                } finally {
                    IamUserTokenHolder.clearContext();
                }
            };
            ExecutorUtil.newExecutor().execute(runnable);
            return this;
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    /**
     * 设置完成
     */
    @Override
    public IShareUploader onCompleted(IUpLoadCallbackable callbackable) {
        this.completeCallback = callbackable;
        return this;
    }
}
