package com.digiwin.dap.middleware.dwpay;

import com.digiwin.dap.middleware.dwpay.common.comm.DefaultServiceClient;
import com.digiwin.dap.middleware.dwpay.common.comm.ServiceClient;
import com.digiwin.dap.middleware.dwpay.internal.DwPayConfig;
import com.digiwin.dap.middleware.dwpay.internal.DwPayConstants;
import com.digiwin.dap.middleware.dwpay.internal.model.TradePeriod;
import com.digiwin.dap.middleware.dwpay.internal.model.TradeParam;
import com.digiwin.dap.middleware.dwpay.internal.operation.TradeOperation;
import com.digiwin.dap.middleware.dwpay.model.AlterType;
import com.digiwin.dap.middleware.dwpay.model.TradeInfo;
import com.digiwin.dap.middleware.dwpay.model.TradeRequest;
import com.digiwin.dap.middleware.dwpay.model.TradeResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * The entry point class of DwPay that implements the DwPay interface.
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class DwPayClient implements DwPay {

    private static final Logger logger = LoggerFactory.getLogger(DwPayClient.class);

    protected ServiceClient serviceClient;
    private TradeOperation tradeOperation;

    DwPayClient(DwPayConfig config) {
        this(config, null);
    }

    DwPayClient(DwPayConfig config, ClientConfiguration clientConfig) {
        DwPayConfig.check(config);
        if (clientConfig == null) {
            this.serviceClient = new DefaultServiceClient(config);
        } else {
            this.serviceClient = new DefaultServiceClient(config, clientConfig);
        }
        initOperations();
    }

    public void initOperations() {
        this.tradeOperation = new TradeOperation(this.serviceClient);
    }

    @Override
    public TradeResult pay(Object data, String userToken) {
        TradeRequest body = new TradeRequest();
        body.setData(data);
        body.addHeader(DwPayConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);
        return tradeOperation.pay(body);
    }

    @Override
    public TradeInfo query(String sourceId, String outTradeNo, String userToken) {
        TradeRequest body = new TradeRequest();
        body.setData(new TradeParam(sourceId, outTradeNo));
        body.addHeader(DwPayConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);
        return tradeOperation.query(body);
    }

    @Override
    public TradeResult close(String sourceId, String outTradeNo, String userToken) {
        TradeRequest body = new TradeRequest();
        body.setData(new TradeParam(sourceId, outTradeNo));
        body.addHeader(DwPayConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);
        return tradeOperation.close(body);
    }

    @Override
    public TradeResult alterStatus(String sourceId, String outTradeNo, AlterType alterType, String userToken) {
        TradeRequest body = new TradeRequest();
        TradePeriod tradePeriod = new TradePeriod();
        tradePeriod.setSourceId(sourceId);
        tradePeriod.setOutTradeNo(outTradeNo);
        tradePeriod.setAlterType(alterType);
        body.setData(tradePeriod);
        body.addHeader(DwPayConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);
        return tradeOperation.alterStatus(body);
    }

    @Override
    public void shutdown() {
        try {
            serviceClient.shutdown();
        } catch (Exception e) {
            logger.error("shutdown throw exception: ", e);
        }
    }

    @Override
    public String getConnectionPoolStats() {
        try {
            return serviceClient.getConnectionPoolStats();
        } catch (Exception ignored) {
        }
        return "";
    }
}
