package com.digiwin.dap.middleware.dwpay;

/**
 * This is the base exception class to represent any expected or unexpected DMC
 * server side errors.
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class ServiceException extends RuntimeException {

    private static final long serialVersionUID = 430933593095358673L;

    private String errorMessage;
    private String errorCode;
    private String requestId;
    private String hostId;

    private String rawResponseError;

    /**
     * Creates a default instance.
     */
    public ServiceException() {
        super();
    }

    /**
     * Creates an instance with the error message.
     *
     * @param errorMessage Error message.
     */
    public ServiceException(String errorMessage) {
        super((String) null);
        this.errorMessage = errorMessage;
    }

    /**
     * Creates an instance with a {@link Throwable} instance.
     *
     * @param cause A {@link Throwable} instance.
     */
    public ServiceException(Throwable cause) {
        super(null, cause);
    }

    /**
     * Creates an instance with a {@link Throwable} instance and error message.
     *
     * @param errorMessage Error message.
     * @param cause        A {@link Throwable} instance.
     */
    public ServiceException(String errorMessage, Throwable cause) {
        super(null, cause);
        this.errorMessage = errorMessage;
    }

    /**
     * Creates an instance with error message, error code, request id, host id.
     *
     * @param errorMessage Error message.
     * @param errorCode    Error code.
     * @param requestId    Request Id.
     * @param hostId       Host Id.
     */
    public ServiceException(String errorMessage, String errorCode, String requestId, String hostId) {
        this(errorMessage, errorCode, requestId, hostId, null);
    }

    /**
     * Creates an instance with error message, error code, request id, host id.
     *
     * @param errorMessage Error message.
     * @param errorCode    Error code.
     * @param requestId    Request Id.
     * @param hostId       Host Id.
     * @param cause        A {@link Throwable} instance indicates a specific exception.
     */
    public ServiceException(String errorMessage, String errorCode, String requestId, String hostId, Throwable cause) {
        this(errorMessage, errorCode, requestId, hostId, null, cause);
    }

    /**
     * Creates an instance with error message, error code, request id, host id,
     * DMC response error, and a Throwable instance.
     *
     * @param errorMessage     Error message.
     * @param errorCode        Error code.
     * @param requestId        Request Id.
     * @param hostId           Host Id.
     * @param rawResponseError DMC error message in response.
     * @param cause            A {@link Throwable} instance indicates a specific exception.
     */
    public ServiceException(String errorMessage, String errorCode, String requestId, String hostId,
                            String rawResponseError, Throwable cause) {
        this(errorMessage, cause);
        this.errorCode = errorCode;
        this.requestId = requestId;
        this.hostId = hostId;
        this.rawResponseError = rawResponseError;
    }

    /**
     * Gets error message.
     *
     * @return Error message.
     */
    public String getErrorMessage() {
        return errorMessage;
    }

    /**
     * Gets the error code.
     *
     * @return The error code in string.
     */
    public String getErrorCode() {
        return errorCode;
    }

    /**
     * Gets the request id.
     *
     * @return The request Id in string.
     */
    public String getRequestId() {
        return requestId;
    }

    /**
     * Gets the host id.
     *
     * @return The host Id in string.
     */
    public String getHostId() {
        return hostId;
    }

    /**
     * Gets the error message in DMC response.
     *
     * @return Error response in string.
     */
    public String getRawResponseError() {
        return rawResponseError;
    }

    /**
     * Sets the error response from DMC.
     *
     * @param rawResponseError The error response from DMC.
     */
    public void setRawResponseError(String rawResponseError) {
        this.rawResponseError = rawResponseError;
    }

    private String formatRawResponseError() {
        if (rawResponseError == null || rawResponseError.equals("")) {
            return "";
        }
        return String.format("\n[ResponseError]:\n%s", this.rawResponseError);
    }

    @Override
    public String getMessage() {
        return getErrorMessage() + "\n[ErrorCode]: " + getErrorCode() + "\n[RequestId]: " + getRequestId()
                + "\n[HostId]: " + getHostId() + formatRawResponseError();
    }
}
