package com.digiwin.dap.middleware.dwpay.common.comm;

import com.digiwin.dap.middleware.dwpay.ClientConfiguration;
import com.digiwin.dap.middleware.dwpay.common.utils.ExceptionFactory;
import com.digiwin.dap.middleware.dwpay.common.utils.HttpHeaders;
import com.digiwin.dap.middleware.dwpay.common.utils.HttpUtil;
import com.digiwin.dap.middleware.dwpay.common.utils.StreamUtils;
import com.digiwin.dap.middleware.dwpay.internal.DwPayConfig;
import com.digiwin.dap.middleware.dwpay.internal.DwPayConstants;
import org.apache.hc.client5.http.classic.methods.HttpUriRequestBase;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.client5.http.io.HttpClientConnectionManager;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.Header;
import org.apache.hc.core5.http.io.SocketConfig;
import org.apache.hc.core5.io.CloseMode;
import org.apache.hc.core5.util.Timeout;

import java.io.IOException;
import java.nio.charset.StandardCharsets;

/**
 * Default implementation of {@link ServiceClient}.
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class DefaultServiceClient extends ServiceClient {

    protected static HttpRequestFactory httpRequestFactory = new HttpRequestFactory();
    protected CloseableHttpClient httpClient;
    protected HttpClientConnectionManager connectionManager;
    protected RequestConfig requestConfig;
    protected DwPayConfig config;

    public DefaultServiceClient(DwPayConfig config) {
        this(config, new ClientConfiguration());
    }

    public DefaultServiceClient(DwPayConfig config, ClientConfiguration clientConfig) {
        super(clientConfig);
        this.connectionManager = createHttpClientConnectionManager();
        this.httpClient = createHttpClient(this.connectionManager);

        RequestConfig.Builder builder = RequestConfig.custom();
        builder.setConnectTimeout(Timeout.ofMilliseconds(clientConfig.getConnectionTimeout()));
        builder.setResponseTimeout(Timeout.ofMilliseconds(clientConfig.getSocketTimeout()));
        builder.setConnectionRequestTimeout(Timeout.ofMilliseconds(clientConfig.getConnectionRequestTimeout()));
        this.requestConfig = builder.build();
        this.config = config;
    }


    private static ResponseMessage buildResponse(RequestMessage request, CloseableHttpResponse httpResponse)
            throws IOException {

        assert (httpResponse != null);

        ResponseMessage response = new ResponseMessage();
        response.setUrl(request.getEndpoint().toString());

        if (httpResponse.getCode() != 0) {
            response.setStatusCode(httpResponse.getCode());
        }

        if (httpResponse.getEntity() != null) {
            response.setContent(httpResponse.getEntity().getContent());
            if (!response.isSuccessful()) {
                String content = StreamUtils.copyToString(response.getContent(), StandardCharsets.UTF_8);
                response.setError(content);
            }
        }

        for (Header header : httpResponse.getHeaders()) {
            if (HttpHeaders.CONTENT_LENGTH.equalsIgnoreCase(header.getName())) {
                response.setContentLength(Long.parseLong(header.getValue()));
            }
            response.addHeader(header.getName(), header.getValue());
        }

        HttpUtil.convertHeaderCharsetFromIso88591(response.getHeaders());

        return response;
    }

    @Override
    public ResponseMessage sendRequestCore(RequestMessage request, ExecutionContext context) throws IOException {
        HttpUriRequestBase httpRequest = httpRequestFactory.createHttpRequest(request);
        handleCommonToken(httpRequest);
        HttpClientContext httpContext = createHttpContext(request);

        CloseableHttpResponse httpResponse;
        try {
            httpResponse = httpClient.execute(httpRequest, httpContext);
        } catch (IOException ex) {
            httpRequest.abort();
            throw ExceptionFactory.createNetworkException(ex);
        }

        return buildResponse(request, httpResponse);
    }

    @Override
    public DwPayConfig getConfig() {
        return config;
    }

    @Override
    public void shutdown() {
        IdleConnectionReaper.remove(this.connectionManager);
        this.connectionManager.close(CloseMode.GRACEFUL);
    }

    @Override
    public String getConnectionPoolStats() {
        if (connectionManager != null && connectionManager instanceof PoolingHttpClientConnectionManager) {
            PoolingHttpClientConnectionManager conn = (PoolingHttpClientConnectionManager) connectionManager;
            return conn.getTotalStats().toString();
        }
        return "";
    }

    private HttpClientConnectionManager createHttpClientConnectionManager() {
        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
        connectionManager.setDefaultMaxPerRoute(clientConfig.getMaxConnections());
        connectionManager.setMaxTotal(clientConfig.getMaxConnections());
        connectionManager.setDefaultSocketConfig(SocketConfig.custom().setSoTimeout(Timeout.ofMilliseconds(clientConfig.getSocketTimeout())).build());
        if (clientConfig.isUseReaper()) {
            IdleConnectionReaper.setIdleConnectionTime(clientConfig.getIdleConnectionTime());
            IdleConnectionReaper.register(connectionManager);
        }
        return connectionManager;
    }

    private CloseableHttpClient createHttpClient(HttpClientConnectionManager connectionManager) {
        return HttpClients.custom().setConnectionManager(connectionManager)
                .disableContentCompression()
                .setRetryStrategy(clientConfig.getRetryHandler())
                .addResponseInterceptorLast(clientConfig.getResponseInterceptor())
                .build();
    }


    private void handleCommonToken(HttpUriRequestBase httpRequest) {
        putHeaderIfAbsent(httpRequest, DwPayConstants.HTTP_HEADER_APP_TOKEN_KEY, config.getAppToken());
    }


    private void putHeaderIfAbsent(HttpUriRequestBase httpRequest, String name, String value) {
        if (!httpRequest.containsHeader(name)) {
            httpRequest.setHeader(name, value);
        }
    }

    private HttpClientContext createHttpContext(RequestMessage request) {
        HttpClientContext httpContext = HttpClientContext.create();
        httpContext.setRequestConfig(this.requestConfig);
        httpContext.setAttribute(CustomHttpRequestRetryHandler.RETRY_ATTRIBUTE, request.isForceRetry());
        return httpContext;
    }
}
