package com.digiwin.dap.middleware.dwpay.common.comm;

import com.digiwin.dap.middleware.dwpay.ClientConfiguration;
import com.digiwin.dap.middleware.dwpay.ClientException;
import com.digiwin.dap.middleware.dwpay.ServiceException;
import com.digiwin.dap.middleware.dwpay.common.utils.CodingUtils;
import com.digiwin.dap.middleware.dwpay.common.utils.DwPayUtils;
import com.digiwin.dap.middleware.dwpay.internal.DwPayConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.List;

/**
 * Abstract service client that provides interfaces to access dwPay services.
 *
 * @author fobgochod
 * @since 1.0.0
 */
public abstract class ServiceClient {

    protected static final Logger logger = LoggerFactory.getLogger(ServiceClient.class);
    protected ClientConfiguration clientConfig;

    protected ServiceClient(ClientConfiguration clientConfig) {
        this.clientConfig = clientConfig;
    }

    public ClientConfiguration getClientConfiguration() {
        return this.clientConfig;
    }

    protected abstract ResponseMessage sendRequestCore(RequestMessage request, ExecutionContext context) throws IOException;

    public abstract DwPayConfig getConfig();


    public abstract void shutdown();

    public String getConnectionPoolStats() {
        return "";
    }

    ;

    /**
     * Send HTTP request with specified context to DMC and wait for HTTP
     * response.
     *
     * @param request 请求信息
     * @param context 上下文
     * @return 响应结果
     */
    public ResponseMessage sendRequest(RequestMessage request, ExecutionContext context)
            throws ServiceException, ClientException {

        CodingUtils.assertParameterNotNull(request, "request");
        CodingUtils.assertParameterNotNull(context, "context");

        try {
            return sendRequestImpl(request, context);
        } finally {
            // Close the request stream as well after the request is completed.
            try {
                request.close();
            } catch (IOException ex) {
                logger.error("Unexpected io exception when trying to close http request: ", ex);
                throw new ClientException("Unexpected io exception when trying to close http request: ", ex);
            }
        }
    }

    public ResponseMessage sendRequestImpl(RequestMessage request, ExecutionContext context) throws ServiceException, ClientException {
        ResponseMessage response = null;
        try {
            // Step 1. Preprocess HTTP request.
            handleRequest(request, context.getRequestHandlers());

            // Step 2. Send HTTP request to DMC.
            response = sendRequestCore(request, context);

            // Step 3. Preprocess HTTP response.
            handleResponse(response, context.getResponseHandlers());

            return response;
        } catch (ServiceException sex) {
            logger.error("[Server]Unable to execute HTTP request: ", sex);
            // Notice that the response should not be closed in the
            // finally block because if the request is successful,
            // the response should be returned to the callers.
            closeResponseSilently(response);
            throw sex;
        } catch (ClientException cex) {
            logger.error("[Client]Unable to execute HTTP request: ", cex);
            closeResponseSilently(response);
            throw cex;
        } catch (Exception ex) {
            logger.error("[Unknown]Unable to execute HTTP request: ", ex);
            closeResponseSilently(response);
            throw new ClientException(DwPayUtils.DW_PAY_RESOURCE_MANAGER.getFormattedString("ConnectionError", ex.getMessage()), ex);
        }
    }

    private void handleRequest(RequestMessage message, List<RequestHandler> requestHandlers)
            throws ServiceException, ClientException {
        for (RequestHandler h : requestHandlers) {
            h.handle(message);
        }
    }

    private void handleResponse(ResponseMessage response, List<ResponseHandler> responseHandlers)
            throws ServiceException, ClientException {
        for (ResponseHandler h : responseHandlers) {
            h.handle(response);
        }
    }

    private void closeResponseSilently(ResponseMessage response) {
        if (response != null) {
            try {
                response.close();
            } catch (IOException ioe) {
                /* silently close the response. */
            }
        }
    }
}


