package com.digiwin.dap.middleware.dwpay.common.utils;

import com.digiwin.dap.middleware.dwpay.ClientErrorCode;
import com.digiwin.dap.middleware.dwpay.ClientException;
import com.digiwin.dap.middleware.dwpay.DwPayErrorCode;
import com.digiwin.dap.middleware.dwpay.DwPayException;
import com.digiwin.dap.middleware.dwpay.internal.model.StdError;
import org.apache.hc.client5.http.ClientProtocolException;
import org.apache.hc.client5.http.ConnectTimeoutException;
import org.apache.hc.client5.http.HttpHostConnectException;
import org.apache.hc.core5.http.NoHttpResponseException;

import java.io.IOException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.nio.channels.NonReadableChannelException;

/**
 * A simple factory used for creating instances of <code>ClientException</code>
 * and <code>DMCException</code>.
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class ExceptionFactory {

    public static ClientException createNetworkException(IOException ex) {
        String requestId = "Unknown";
        String errorCode = ClientErrorCode.UNKNOWN;

        if (ex instanceof SocketTimeoutException) {
            errorCode = ClientErrorCode.SOCKET_TIMEOUT;
        } else if (ex instanceof SocketException) {
            errorCode = ClientErrorCode.SOCKET_EXCEPTION;
        } else if (ex instanceof ConnectTimeoutException) {
            errorCode = ClientErrorCode.CONNECTION_TIMEOUT;
        } else if (ex instanceof UnknownHostException) {
            errorCode = ClientErrorCode.UNKNOWN_HOST;
        } else if (ex instanceof HttpHostConnectException) {
            errorCode = ClientErrorCode.CONNECTION_REFUSED;
        } else if (ex instanceof NoHttpResponseException) {
            errorCode = ClientErrorCode.CONNECTION_TIMEOUT;
        } else if (ex instanceof ClientProtocolException) {
            Throwable cause = ex.getCause();
            if (cause instanceof NonReadableChannelException) {
                errorCode = ClientErrorCode.NONREPEATABLE_REQUEST;
                return new ClientException(cause.getMessage(), errorCode, requestId, cause);
            }
        }
        return new ClientException(ex.getMessage(), errorCode, requestId, ex);
    }

    public static DwPayException createInvalidResponseException(String requestId, String rawResponseError,
                                                                Throwable cause) {
        return createInvalidResponseException(requestId,
                DwPayUtils.DW_PAY_RESOURCE_MANAGER.getFormattedString("FailedToParseResponse", cause.getMessage()),
                rawResponseError);
    }

    public static DwPayException createInvalidResponseException(String requestId, String message,
                                                                String rawResponseError) {
        return createDMCException(requestId, DwPayErrorCode.INVALID_RESPONSE, message, rawResponseError);
    }

    public static DwPayException createDMCException(String requestId, String errorCode, String message,
                                                    String rawResponseError) {
        return new DwPayException(message, errorCode, requestId, null, null, null, null, rawResponseError);
    }

    public static DwPayException createDMCException(String requestId, String errorCode, String message) {
        return new DwPayException(message, errorCode, requestId, null, null, null, null);
    }

    public static DwPayException createUnknownDMCException(String requestId, int statusCode) {
        String message = "No body in response, http status code " + statusCode;
        return new DwPayException(message, ClientErrorCode.UNKNOWN, requestId, null, null, null, null);
    }

    public static DwPayException createDMCException(StdError errorResult) {
        return new DwPayException(String.format("%s：%s", errorResult.getErrorMessage(), errorResult.getMessage()),
                errorResult.getErrorCode());
    }
}
