package com.digiwin.dap.middleware.dwpay.internal;

import com.digiwin.dap.middleware.dwpay.ClientException;
import com.digiwin.dap.middleware.dwpay.DwPayException;
import com.digiwin.dap.middleware.dwpay.ServiceException;
import com.digiwin.dap.middleware.dwpay.common.comm.*;
import com.digiwin.dap.middleware.dwpay.common.parser.ResponseParseException;
import com.digiwin.dap.middleware.dwpay.common.parser.ResponseParser;
import com.digiwin.dap.middleware.dwpay.common.utils.DwPayUtils;
import com.digiwin.dap.middleware.dwpay.common.utils.ExceptionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Abstract base class that provides some common functionalities for DMC
 * operations (such as bucket/file/multipart operations).
 *
 * @author fobgochod
 * @since 1.0.0
 */
public abstract class DwPayOperation {

    protected static final Logger logger = LoggerFactory.getLogger(DwPayOperation.class);
    protected static ResponseHandler errorResponseHandler = new ErrorResponseHandler();
    protected ServiceClient client;
    protected DwPayConfig config;

    protected DwPayOperation(ServiceClient client) {
        this.client = client;
        this.config = client.getConfig();
    }

    protected <T> T doOperation(RequestMessage request, ResponseParser<T> parser) {
        return doOperation(request, parser, true);
    }

    protected <T> T doOperation(RequestMessage request, ResponseParser<T> parser, boolean keepResponseOpen) {
        ExecutionContext context = createDefaultContext();
        ResponseMessage response = this.send(request, context, keepResponseOpen);
        try {
            response.setAppSecret(config.getAppSecret());
            return parser.parse(response);
        } catch (ResponseParseException rpe) {
            DwPayException de = ExceptionFactory.createInvalidResponseException(response.getRequestId(), rpe.getMessage(), rpe);
            logger.error("Unable to parse response error: ", rpe);
            throw de;
        }
    }

    protected ResponseMessage send(RequestMessage request, ExecutionContext context, boolean keepResponseOpen)
            throws ServiceException, ClientException {
        ResponseMessage response = null;
        try {
            response = client.sendRequest(request, context);
            return response;
        } finally {
            if (response != null && !keepResponseOpen) {
                DwPayUtils.safeCloseResponse(response);
            }
        }
    }

    protected ExecutionContext createDefaultContext() {
        ExecutionContext context = new ExecutionContext();
        context.addResponseHandler(errorResponseHandler);
        return context;
    }
}
