package com.digiwin.dap.middleware.dwpay.internal;

import com.digiwin.dap.middleware.dwpay.ClientException;
import com.digiwin.dap.middleware.dwpay.DwPayErrorCode;
import com.digiwin.dap.middleware.dwpay.DwPayException;
import com.digiwin.dap.middleware.dwpay.common.comm.ResponseHandler;
import com.digiwin.dap.middleware.dwpay.common.comm.ResponseMessage;
import com.digiwin.dap.middleware.dwpay.common.parser.ErrorResponseParser;
import com.digiwin.dap.middleware.dwpay.common.parser.ResponseParseException;
import com.digiwin.dap.middleware.dwpay.common.utils.DwPayUtils;
import com.digiwin.dap.middleware.dwpay.common.utils.ExceptionFactory;
import com.digiwin.dap.middleware.dwpay.internal.model.StdError;
import org.apache.hc.core5.http.HttpStatus;

/**
 * error response handler
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class ErrorResponseHandler implements ResponseHandler {

    @Override
    public void handle(ResponseMessage response) throws DwPayException, ClientException {

        if (response.isSuccessful()) {
            return;
        }

        String requestId = response.getRequestId();
        int statusCode = response.getStatusCode();
        if (response.getContent() == null) {
            /**
             * When HTTP response body is null, handle status code 404 Not
             * Found, 304 Not Modified, 412 Precondition Failed especially.
             */
            if (statusCode == HttpStatus.SC_NOT_FOUND) {
                throw ExceptionFactory.createDMCException(requestId, DwPayErrorCode.NO_SUCH_KEY, "Not Found");
            } else if (statusCode == HttpStatus.SC_NOT_MODIFIED) {
                throw ExceptionFactory.createDMCException(requestId, DwPayErrorCode.NOT_MODIFIED, "Not Modified");
            } else if (statusCode == HttpStatus.SC_PRECONDITION_FAILED) {
                throw ExceptionFactory.createDMCException(requestId, DwPayErrorCode.PRECONDITION_FAILED, "Precondition Failed");
            } else if (statusCode == HttpStatus.SC_FORBIDDEN) {
                throw ExceptionFactory.createDMCException(requestId, DwPayErrorCode.ACCESS_FORBIDDEN, "AccessForbidden");
            } else {
                throw ExceptionFactory.createUnknownDMCException(requestId, statusCode);
            }
        }

        try {
            ErrorResponseParser parser = new ErrorResponseParser();
            StdError stdError = parser.parse(response);
            throw ExceptionFactory.createDMCException(stdError);
        } catch (ResponseParseException e) {
            throw ExceptionFactory.createInvalidResponseException(requestId, response.getError(), e);
        } finally {
            DwPayUtils.safeCloseResponse(response);
        }
    }
}
