package com.digiwin.dap.middleware.emc;

import com.digiwin.dap.middleware.emc.common.comm.DefaultServiceClient;
import com.digiwin.dap.middleware.emc.common.comm.ServiceClient;
import com.digiwin.dap.middleware.emc.internal.EMCConfig;
import com.digiwin.dap.middleware.emc.internal.EMCHeaders;
import com.digiwin.dap.middleware.emc.internal.operation.EMCBaseOperation;
import com.digiwin.dap.middleware.emc.model.MessageInfo;
import com.digiwin.dap.middleware.emc.model.MessageInfoRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;

/**
 * The entry point class of EMC that implements the EMC interface.
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class EMCClient implements EMC {

    private static final Logger logger = LoggerFactory.getLogger(EMCClient.class);
    private final EMCBaseOperation baseOperation;
    protected ServiceClient serviceClient;

    EMCClient(EMCConfig config) {
        this(config, null);
    }

    EMCClient(EMCConfig config, ClientConfiguration clientConfig) {
        if (clientConfig == null) {
            this.serviceClient = new DefaultServiceClient(config);
        } else {
            this.serviceClient = new DefaultServiceClient(config, clientConfig);
        }
        this.baseOperation = new EMCBaseOperation(this.serviceClient);
    }

    @Override
    public void send(String eventId, Object payload, String userToken) {
        send(eventId, payload, null, userToken);
    }

    @Override
    public void send(String eventId, Object payload, String contacts, String userToken) {
        MessageInfo messageInfo = new MessageInfo();
        messageInfo.setEventId(eventId);
        messageInfo.setContacts(contacts);
        messageInfo.setMessage(Collections.singletonMap("data", payload));
        send(messageInfo, userToken);
    }

    @Override
    public void send(MessageInfo message, String userToken) {
        MessageInfoRequest request = new MessageInfoRequest();
        request.setMessageInfo(message);
        request.addHeader(EMCHeaders.HTTP_HEADER_USER_TOKEN_KEY, userToken);
        baseOperation.send(request);
    }

    @Override
    public void shutdown() {
        try {
            serviceClient.shutdown();
        } catch (Exception e) {
            logger.error("shutdown throw exception: ", e);
        }
    }

    @Override
    public String getConnectionPoolStats() {
        try {
            return serviceClient.getConnectionPoolStats();
        } catch (Exception ignored) {
        }
        return "";
    }
}
