package com.digiwin.dap.middleware.emc.internal;

import com.digiwin.dap.middleware.emc.ClientException;
import com.digiwin.dap.middleware.emc.EMCErrorCode;
import com.digiwin.dap.middleware.emc.EMCException;
import com.digiwin.dap.middleware.emc.common.comm.ResponseHandler;
import com.digiwin.dap.middleware.emc.common.comm.ResponseMessage;
import com.digiwin.dap.middleware.emc.common.parser.ErrorResponseParser;
import com.digiwin.dap.middleware.emc.common.parser.ResponseParseException;
import com.digiwin.dap.middleware.emc.common.utils.ExceptionFactory;
import com.digiwin.dap.middleware.emc.internal.model.StdError;
import org.apache.hc.core5.http.HttpStatus;

/**
 * error response handler
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class ErrorResponseHandler implements ResponseHandler {

    @Override
    public void handle(ResponseMessage response) throws EMCException, ClientException {

        if (response.isSuccessful()) {
            return;
        }

        String requestId = response.getRequestId();
        int statusCode = response.getStatusCode();
        if (response.getContent() == null) {
            /**
             * When HTTP response body is null, handle status code 404 Not
             * Found, 304 Not Modified, 412 Precondition Failed especially.
             */
            if (statusCode == HttpStatus.SC_NOT_FOUND) {
                throw ExceptionFactory.createEMCException(requestId, EMCErrorCode.NO_SUCH_KEY, "Not Found");
            } else if (statusCode == HttpStatus.SC_NOT_MODIFIED) {
                throw ExceptionFactory.createEMCException(requestId, EMCErrorCode.NOT_MODIFIED, "Not Modified");
            } else if (statusCode == HttpStatus.SC_PRECONDITION_FAILED) {
                throw ExceptionFactory.createEMCException(requestId, EMCErrorCode.PRECONDITION_FAILED, "Precondition Failed");
            } else if (statusCode == HttpStatus.SC_FORBIDDEN) {
                throw ExceptionFactory.createEMCException(requestId, EMCErrorCode.ACCESS_FORBIDDEN, "AccessForbidden");
            } else {
                throw ExceptionFactory.createUnknownEMCException(requestId, statusCode);
            }
        }

        try {
            ErrorResponseParser parser = new ErrorResponseParser();
            StdError stdError = parser.parse(response);
            throw ExceptionFactory.createEMCException(stdError);
        } catch (ResponseParseException e) {
            throw ExceptionFactory.createInvalidResponseException(requestId, response.getError(), e);
        } finally {
            EMCUtils.safeCloseResponse(response);
        }
    }
}
