package com.digiwin.dap.middleware.lmc.common.comm;

import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.core5.util.TimeValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;

/**
 * A daemon thread used to periodically check connection pools for idle
 * connections. 使用 HttpClient 5.x 的 PoolingHttpClientConnectionManager 实现并优化并发访问。
 * <p>
 * 优化点：
 * - 使用 CopyOnWriteArrayList 减少同步开销，在迭代时无需再 clone 列表。
 * - 调用 5.x 的 closeExpired() 和 closeIdle(TimeValue) 方法。
 *
 * @author chenzhuang
 * @date 2021/4/23
 */
public final class IdleConnectionReaper extends Thread {

    private static final Logger logger = LoggerFactory.getLogger(IdleConnectionReaper.class);
    private static final int REAP_INTERVAL_MILLISECONDS = 5 * 1000;
    private static final CopyOnWriteArrayList<PoolingHttpClientConnectionManager> connectionManagers = new CopyOnWriteArrayList<>();

    private static IdleConnectionReaper instance;

    private static volatile long idleConnectionTime = 60 * 1000;

    private volatile boolean shuttingDown;

    private IdleConnectionReaper() {
        super("idle-connection-reaper");
        setDaemon(true);
    }

    public static synchronized boolean registerConnectionManager(PoolingHttpClientConnectionManager connectionManager) {
        if (instance == null) {
            instance = new IdleConnectionReaper();
            instance.start();
        }
        return connectionManagers.addIfAbsent(connectionManager);
    }

    public static synchronized boolean removeConnectionManager(PoolingHttpClientConnectionManager connectionManager) {
        boolean removed = connectionManagers.remove(connectionManager);
        if (connectionManagers.isEmpty()) {
            shutdown();
        }
        return removed;
    }

    public static synchronized boolean shutdown() {
        if (instance != null) {
            instance.markShuttingDown();
            instance.interrupt();
            connectionManagers.clear();
            instance = null;
            return true;
        }
        return false;
    }

    public static int size() {
        return connectionManagers.size();
    }

    public static void setIdleConnectionTime(long idletime) {
        idleConnectionTime = idletime;
    }

    private void markShuttingDown() {
        shuttingDown = true;
    }

    @SuppressWarnings("unchecked")
    @Override
    public void run() {
        while (true) {
            if (shuttingDown) {
                logger.debug("Shutting down reaper thread.");
                return;
            }

            try {
                Thread.sleep(REAP_INTERVAL_MILLISECONDS);
            } catch (InterruptedException e) {
            }

            try {
                List<PoolingHttpClientConnectionManager> managersSnapshot = connectionManagers;
                for (PoolingHttpClientConnectionManager connectionManager : managersSnapshot) {
                    try {
                        connectionManager.closeExpired();
                        connectionManager.closeIdle(TimeValue.of(idleConnectionTime, TimeUnit.MILLISECONDS));
                    } catch (Exception ex) {
                        logger.warn("Unable to close idle connections", ex);
                    }
                }
            } catch (Throwable t) {
                logger.debug("Reaper thread: ", t);
            }
        }
    }
}