package com.digiwin.dap.middleware.lmc.appender;

import com.digiwin.dap.middleware.lmc.http.client.HttpConfig;
import com.digiwin.dap.middleware.lmc.internal.LMCResourceUri;
import com.digiwin.dap.middleware.lmc.util.LogUtils;
import com.digiwin.dap.middleware.lmc.util.LoggingEventSizeUtil;
import com.digiwin.dap.middleware.lmc.util.ThreadPoolUtil;
import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.io.entity.StringEntity;
import org.apache.hc.core5.util.Timeout;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.layout.PatternLayout;
import org.apache.logging.log4j.core.util.Throwables;
import org.apache.logging.log4j.util.PropertiesUtil;
import org.json.JSONArray;

import java.io.IOException;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * DwLogAppender
 *
 * @author chenzhuang
 */
@Plugin(
        name = "DwLog4j2Appender",
        category = "Core",
        elementType = "appender",
        printObject = true
)
public class DwLog4j2Appender extends AbstractAppender {
    static final String PROPERTY_NAME_ASYNC_EVENT_ROUTER = "log4j2.AsyncQueueFullPolicy";
    static final String PROPERTY_VALUE_DISCARDING_ASYNC_EVENT_ROUTER = "Discard";
    private final SimpleDateFormat _sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

    private static final String CUSTOM_CONTENT_KEY_SEPARATOR = ",";

    private static List<String> CUSTOM_CONTENT_KEY_LIST = new ArrayList<>();

    /**
     * 默认间隔时间:2s
     */
    private static final int DEFAULT_PERIOD_MILL_SECONDS = 2;
    /**
     * 默认批量条数:50
     */
    private static final int DEFAULT_BATCH_SIZE = 50;
    /**
     * 默认单条日志最大大小:10KB
     */
    private static final Long DEFAULT_MAX_LOG_SIZE = 10240L;

    private static final Integer DEFAULT_THREAD_NUM = 5;

    private ScheduledExecutorService executor;

    private int consumeCount = 0;

    private Queue<Map<String, Object>> workQueue = new LinkedList<>();

    private CloseableHttpClient client;

    private Integer intervals;

    private Integer batchSize;

    private Long maxSingleLogSize;

    private String customContentKey;

    private String endpoint;
    private String app;
    private String userAgent = "log4j2";

    protected DwLog4j2Appender(String name, Filter filter, Layout<? extends Serializable> layout, String endpoint,
                               String app, Integer intervals, Integer batchSize, Long maxSingleLogSize, String customContentKey) {
        super(name, filter, layout);
        this.endpoint = endpoint;
        this.app = app;
        this.intervals = intervals;
        this.batchSize = batchSize;
        this.maxSingleLogSize = maxSingleLogSize;
        this.customContentKey = customContentKey;
    }

    @Override
    public void start() {
        super.start();
        // 请求配置
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(Timeout.ofMilliseconds(HttpConfig.CONNECT_TIMEOUT))
                .setConnectionRequestTimeout(Timeout.ofMilliseconds(HttpConfig.CONNECT_REQUEST_TIMEOUT))
                .setResponseTimeout(Timeout.ofMilliseconds(HttpConfig.SOCKET_TIMEOUT))
                .build();

        // 连接池管理
        PoolingHttpClientConnectionManager cm = new PoolingHttpClientConnectionManager();
        cm.setMaxTotal(HttpConfig.MAX_TOTAL_CONNECTIONS);
        cm.setDefaultMaxPerRoute(HttpConfig.MAXIMUM_CONNECTION_PER_ROUTE);
        cm.setValidateAfterInactivity(Timeout.ofMilliseconds(HttpConfig.CONNECTION_VALIDATE_AFTER_INACTIVITY_MS));

        this.client = HttpClients.custom()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionManager(cm).build();

        // 初始化配置
        if (PropertiesUtil.getProperties().getStringProperty(PROPERTY_NAME_ASYNC_EVENT_ROUTER) == null) {
            System.setProperty(PROPERTY_NAME_ASYNC_EVENT_ROUTER, PROPERTY_VALUE_DISCARDING_ASYNC_EVENT_ROUTER);
        }

        // 配置定时
        executor = Executors.newSingleThreadScheduledExecutor();
        executor.scheduleAtFixedRate(() -> {
            synchronized (DwLog4j2Appender.this) {
                if (!workQueue.isEmpty()) {
                    List<Map<String, Object>> logList = new LinkedList<>();
                    for (Map<String, Object> element : workQueue) {
                        logList.add(element);
                    }
                    asyncPersistence(logList);
                    consumeCount = 0;
                    workQueue.clear();
                }
            }
        }, 0, this.intervals, TimeUnit.SECONDS);

        super.start();
    }

    @Override
    public void stop() {
        if (!isStarted()) {
            return;
        }
        List<Map<String, Object>> logList = new LinkedList<>();
        for (Map<String, Object> element : workQueue) {
            logList.add(element);
        }
        asyncPersistence(logList);
        executor.shutdown();
        super.stop();
        if (this.client != null) {
            try {
                this.client.close();
            } catch (IOException exception) {
                LOGGER.error(exception.getMessage());
            }
        }
    }

    @Override
    public void append(LogEvent event) {
        synchronized (this) {
            // 队列数据超出单次批量提交数量,说明当前生产者生产数据消费者未及时消费,阻塞线程,等待消费者消费
            while (this.workQueue.size() > this.batchSize) {
                try {
                    wait();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return;
                }
            }
            Map<String, Object> map = convertLogEventToMap(event);
            if (Objects.nonNull(map)) {
                if (LoggingEventSizeUtil.getSize(map) >= this.maxSingleLogSize * 1024) {
                    // 单条日志超出1MB直接提交,不进入队列不计数且此处不清空队列,清空队列会导致未提交数据丢失
                    List<Map<String, Object>> log = Arrays.asList(map);
                    asyncPersistence(log);
                } else {
                    // 将新生产的数据放入队列,等待消费
                    this.workQueue.offer(map);
                    consumeCount++;
                }
            }

            // 满足消费条件,消费队列数据
            if (consumeCount >= this.batchSize) {
                List<Map<String, Object>> logList = new LinkedList<>();
                for (Map<String, Object> element : workQueue) {
                    logList.add(element);
                }
                asyncPersistence(logList);
                this.workQueue.clear();
                consumeCount = 0;
                // 队列数据被消费,唤醒等待的线程
                notifyAll();
            }
        }
    }

    private Map<String, Object> convertLogEventToMap(LogEvent event) {
        Map<String, Object> logMap = new HashMap();
        Map<String, String> contextDataMap = event.getContextData().toMap();
        String traceId;
        if (!contextDataMap.isEmpty()) {
            String ptxId = contextDataMap.get("PtxId");
            // String pspanId = contextDataMap.get("PspanId ");
            if (ptxId != null && !"".equals(ptxId)) {
                traceId = ptxId;
            } else {
                traceId = LogUtils.getUUID();
            }
            Map<String, String> customContent = new HashMap<>();
            CUSTOM_CONTENT_KEY_LIST.forEach(key -> {
                if (Objects.nonNull(contextDataMap.get(key))) {
                    customContent.put(key, contextDataMap.get(key));
                }
            });
            logMap.put("customContent", customContent);
        } else {
            traceId = LogUtils.getUUID();
        }
        logMap.putIfAbsent("traceId", traceId);

        Map<String, Object> initMap = new HashMap<>();
        LogUtils.initLogMap(initMap);
        logMap.putAll(initMap);
        logMap.putIfAbsent("appId", this.getApp());
        logMap.putIfAbsent("time", this._sdf.format(new Date(event.getTimeMillis())));
        logMap.putIfAbsent("level", event.getLevel().toString());
        logMap.putIfAbsent("thread", event.getThreadName());

        logMap.putIfAbsent("loggerName", event.getLoggerName());
        String addr = LogUtils.getLocalHostIpName();
        logMap.putIfAbsent("source", addr);

        logMap.putIfAbsent("appender", "DwLog4j2Appender");

        StackTraceElement source = event.getSource();
        if (source == null && (!event.isIncludeLocation())) {
            event.setIncludeLocation(true);
            source = event.getSource();
            event.setIncludeLocation(false);
        }

        logMap.putIfAbsent("location", source == null ? "Unknown(Unknown Source)" : source.toString());

        String message = event.getMessage().getFormattedMessage();
        logMap.putIfAbsent("message", message);

        if (Objects.nonNull(event.getThrown())) {
            StringBuilder sb = new StringBuilder();
            boolean isFirst = true;
            for (String s : Throwables.toStringList(event.getThrown())) {
                if (isFirst) {
                    isFirst = false;
                } else {
                    sb.append(System.getProperty("line.separator"));
                }
                sb.append(s);
            }
            logMap.putIfAbsent("throwable", sb.toString());
        }

        if (getLayout() != null) {
            logMap.putIfAbsent("log", new String(getLayout().toByteArray(event)));
        }
        return logMap;
    }

    private void asyncPersistence(List<Map<String, Object>> logList) {
        ThreadPoolUtil.executor(() -> {
            if (logList == null || logList.isEmpty()) {
                return;
            }

            HttpPost post = new HttpPost(LMCResourceUri.getSaveDevLogBatchUrl(this.endpoint));
            try {
                JSONArray jsonArray = new JSONArray(logList);
                post.setEntity(new StringEntity(jsonArray.toString(), ContentType.APPLICATION_JSON));

                ClassicHttpResponse response = this.client.execute(post, httpResponse -> httpResponse);

                if (response.getCode() != 200) {
                    post.cancel();
                }
            } catch (Exception e) {
                LOGGER.error(e.getMessage());
            }
        });
    }

    @PluginFactory
    public static DwLog4j2Appender createAppender(
            @PluginAttribute("name") String name,
            @PluginElement("Filter") final Filter filter,
            @PluginElement("Layout") Layout<? extends Serializable> layout,
            @PluginAttribute("endpoint") String endpoint,
            @PluginAttribute("app") String app,
            @PluginAttribute("intervals") Integer intervals,
            @PluginAttribute("batchSize") Integer batchSize,
            @PluginAttribute("maxSingleLogSize") Long maxSingleLogSize,
            @PluginAttribute("customContentKey") String customContentKey) {
        if (name == null) {
            LOGGER.error("no name defined in conf.");
            return null;
        }
        if (layout == null) {
            layout = PatternLayout.createDefaultLayout();
        }
        if(0 == intervals){
            intervals = DEFAULT_PERIOD_MILL_SECONDS;
        }
        if(0 == batchSize){
            batchSize = DEFAULT_BATCH_SIZE;
        }
        if(Objects.equals(0L, maxSingleLogSize)){
            maxSingleLogSize = DEFAULT_MAX_LOG_SIZE;
        }
        if(null != customContentKey && customContentKey.length() != 0){
            CUSTOM_CONTENT_KEY_LIST = Arrays.asList(customContentKey.split(CUSTOM_CONTENT_KEY_SEPARATOR));
        }
        return new DwLog4j2Appender(name, filter, (Layout) layout, endpoint, app, intervals, batchSize, maxSingleLogSize, customContentKey);
    }

    public String getEndpoint() {
        return this.endpoint;
    }

    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    public String getApp() {
        return this.app;
    }

    public void setApp(String app) {
        this.app = app;
    }

    public CloseableHttpClient getHttpClient() {
        return this.client;
    }

    public void setHttpClient(CloseableHttpClient client) {
        this.client = client;
    }

    public Integer getIntervals() {
        return intervals;
    }

    public void setIntervals(Integer intervals) {
        this.intervals = intervals;
    }

    public Integer getBatchSize() {
        return batchSize;
    }

    public void setBatchSize(Integer batchSize) {
        this.batchSize = batchSize;
    }

    public Long getMaxSingleLogSize() {
        return maxSingleLogSize;
    }

    public void setMaxSingleLogSize(Long maxSingleLogSize) {
        this.maxSingleLogSize = maxSingleLogSize;
    }

    public String getCustomContentKey() {
        return customContentKey;
    }

    public void setCustomContentKey(String customContentKey) {
        this.customContentKey = customContentKey;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public void setUserAgent(String userAgent) {
        this.userAgent = userAgent;
    }
}
