package com.digiwin.dap.middleware.lmc.appender;

import ch.qos.logback.classic.spi.IThrowableProxy;
import ch.qos.logback.classic.spi.LoggingEvent;
import ch.qos.logback.classic.spi.StackTraceElementProxy;
import ch.qos.logback.classic.spi.ThrowableProxyUtil;
import ch.qos.logback.core.CoreConstants;
import ch.qos.logback.core.UnsynchronizedAppenderBase;
import ch.qos.logback.core.encoder.Encoder;
import com.digiwin.dap.middleware.lmc.internal.LMCResourceUri;
import com.digiwin.dap.middleware.lmc.util.LogUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.json.JSONObject;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * LoghubAppender
 *
 * @author chenzhuang
 */
public class DwLogbackAppender<E> extends UnsynchronizedAppenderBase<E> {
    private SimpleDateFormat _sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
    private CloseableHttpClient client;

    private String app;
    private String endpoint;
    protected Encoder<E> encoder;
    private String userAgent = "logback";

    @Override
    public void start() {
        try {
            doStart();
        } catch (Exception e) {
            addError("Failed to start LoghubAppender.", e);
        }
    }

    private void doStart() {
        super.start();
        this.endpoint = endpoint + LMCResourceUri.DEVLOG;

        RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(5000).setConnectionRequestTimeout(5000).build();
        this.client = HttpClientBuilder.create().setDefaultRequestConfig(requestConfig).build();
    }

    @Override
    public void stop() {
        try {
            doStop();
        } catch (Exception e) {
            addError("Failed to stop LoghubAppender.", e);
        }
    }

    private void doStop() {
        if (!isStarted()) {
            return;
        }
        super.stop();

        if (this.client != null) {
            try {
                this.client.close();
            } catch (IOException var2) {
                addError(var2.getMessage());
            }
        }
    }

    @Override
    public void append(E eventObject) {
        try {
            appendEvent(eventObject);
        } catch (Exception e) {
            addError("Failed to append event.", e);
        }
    }

    private void appendEvent(E eventObject) {
        CloseableHttpResponse h_response = null;

        //init Event Object
        if (!(eventObject instanceof LoggingEvent)) {
            return;
        }
        LoggingEvent event = (LoggingEvent) eventObject;
        HttpPost postMethod = new HttpPost(this.getEndpoint());
        Map<String, Object> logMap = new HashMap();

        Map<String, String> mdcPropertyMap = event.getMDCPropertyMap();
        String traceId;
        if (!mdcPropertyMap.isEmpty()) {
            String ptxId = mdcPropertyMap.get("PtxId");
            //String pspanId = mdcPropertyMap.get("PspanId ");
            if (ptxId != null && !"".equals(ptxId)) {
                traceId = ptxId;
            } else {
                traceId = LogUtils.getUUID();
            }
        } else {
            traceId = LogUtils.getUUID();
        }
        logMap.putIfAbsent("traceId", traceId);

        Map<String, Object> initMap = new HashMap<>();
        LogUtils.initLogMap(initMap);
        logMap.putAll(initMap);
        logMap.putIfAbsent("appId", this.getApp());
        logMap.putIfAbsent("time", this._sdf.format(new Date()));
        logMap.putIfAbsent("level", event.getLevel().toString());
        logMap.putIfAbsent("thread", event.getThreadName());

        logMap.putIfAbsent("loggerName", event.getLoggerName());
        String addr = LogUtils.getLocalHostIpName();
        logMap.putIfAbsent("source", addr);

        logMap.putIfAbsent("appender", "DwLogbackAppender");

        StackTraceElement[] caller = event.getCallerData();
        if (caller != null && caller.length > 0) {
            logMap.putIfAbsent("location", caller[0].toString());
        }

        String message = event.getFormattedMessage();
        logMap.putIfAbsent("message", message);

        IThrowableProxy iThrowableProxy = event.getThrowableProxy();
        if (iThrowableProxy != null) {
            String throwable = getExceptionInfo(iThrowableProxy);
            throwable += fullDump(event.getThrowableProxy().getStackTraceElementProxyArray());
            logMap.putIfAbsent("throwable", throwable);
        }

        if (this.encoder != null) {
            logMap.putIfAbsent("log", new String(this.encoder.encode(eventObject)));
        }

        JSONObject tmepObject = new JSONObject(logMap);
        String formatValue = tmepObject.toString(5);
        postMethod.setEntity(new StringEntity(formatValue, ContentType.APPLICATION_JSON));
        try {
            h_response = this.client.execute(postMethod);
            HttpEntity repEntity = h_response.getEntity();
            int statusCode = h_response.getStatusLine().getStatusCode();
            if (statusCode != 200) {
                postMethod.abort();
            }
        } catch (Exception e) {
            addError(e.getMessage());
        } finally {
            if (h_response != null) {
                try {
                    h_response.close();
                } catch (IOException e) {
                    addError(e.getMessage());
                }
            }

        }
    }

    private String getExceptionInfo(IThrowableProxy iThrowableProxy) {
        String s = iThrowableProxy.getClassName();
        String message = iThrowableProxy.getMessage();
        return (message != null) ? (s + ": " + message) : s;
    }

    private String fullDump(StackTraceElementProxy[] stackTraceElementProxyArray) {
        StringBuilder builder = new StringBuilder();
        for (StackTraceElementProxy step : stackTraceElementProxyArray) {
            builder.append(CoreConstants.LINE_SEPARATOR);
            String string = step.toString();
            builder.append(CoreConstants.TAB).append(string);
            ThrowableProxyUtil.subjoinPackagingData(builder, step);
        }
        return builder.toString();
    }

    public String getApp() {
        return app;
    }

    public void setApp(String app) {
        this.app = app;
    }

    public String getEndpoint() {
        return endpoint;
    }

    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public void setUserAgent(String userAgent) {
        this.userAgent = userAgent;
    }

    public Encoder<E> getEncoder() {
        return encoder;
    }

    public void setEncoder(Encoder<E> encoder) {
        this.encoder = encoder;
    }
}
