package com.digiwin.dap.middleware.lmc.http.client;

/**
 * <p>
 * 表示服务器返回的错误消息。
 * </p>
 *

 * <p>
 * {@link ServiceException}用于处理服务器返回的错误消息。比如，用于身份验证的Access ID不存在，
 * 则会抛出{@link ServiceException}（严格上讲，会是该类的一个继承类。比如，OTSClient会抛出OTSException）。
 * 异常中包含了错误代码，用于让调用者进行特定的处理。
 * </p>
 *
 * <p>
 * {@link ClientException}表示的则是在向服务器发送请求时出现的错误，以及客户端无法处理返回结果。
 * 例如，在发送请求时网络连接不可用，则会抛出{@link ClientException}的异常。
 * </p>
 *
 * <p>
 * 通常来讲，调用者只需要处理{@link ServiceException}。因为该异常表明请求被服务处理，但处理的结果表明
 * 存在错误。异常中包含了细节的信息，特别是错误代码，可以帮助调用者进行处理。
 * </p>
 *
 * @author chenzhuang
 * @date 2021/4/23
 */
public class ServiceException extends RuntimeException {

    private static final long serialVersionUID = 430933593095358673L;

    private String errorMessage;
    private String errorCode;
    private String requestId;
    private String hostId;

    private String rawResponseError;

    /**
     * Creates a default instance.
     */
    public ServiceException() {
        super();
    }

    /**
     * Creates an instance with the error message.
     *
     * @param errorMessage Error message.
     */
    public ServiceException(String errorMessage) {
        super((String) null);
        this.errorMessage = errorMessage;
    }

    /**
     * Creates an instance with a {@link Throwable} instance.
     *
     * @param cause A {@link Throwable} instance.
     */
    public ServiceException(Throwable cause) {
        super(null, cause);
    }

    /**
     * Creates an instance with a {@link Throwable} instance and error message.
     *
     * @param errorMessage Error message.
     * @param cause        A {@link Throwable} instance.
     */
    public ServiceException(String errorMessage, Throwable cause) {
        super(null, cause);
        this.errorMessage = errorMessage;
    }

    /**
     * Creates an instance with error message, error code, request id, host id.
     *
     * @param errorMessage Error message.
     * @param errorCode    Error code.
     * @param requestId    Request Id.
     * @param hostId       Host Id.
     */
    public ServiceException(String errorMessage, String errorCode, String requestId, String hostId) {
        this(errorMessage, errorCode, requestId, hostId, null);
    }

    /**
     * Creates an instance with error message, error code, request id, host id.
     *
     * @param errorMessage Error message.
     * @param errorCode    Error code.
     * @param requestId    Request Id.
     * @param hostId       Host Id.
     * @param cause        A {@link Throwable} instance indicates a specific exception.
     */
    public ServiceException(String errorMessage, String errorCode, String requestId, String hostId, Throwable cause) {
        this(errorMessage, errorCode, requestId, hostId, null, cause);
    }

    /**
     * Creates an instance with error message, error code, request id, host id,
     * OSS response error, and a Throwable instance.
     *
     * @param errorMessage     Error message.
     * @param errorCode        Error code.
     * @param requestId        Request Id.
     * @param hostId           Host Id.
     * @param rawResponseError OSS error message in response.
     * @param cause            A {@link Throwable} instance indicates a specific exception.
     */
    public ServiceException(String errorMessage, String errorCode, String requestId, String hostId,
                            String rawResponseError, Throwable cause) {
        this(errorMessage, cause);
        this.errorCode = errorCode;
        this.requestId = requestId;
        this.hostId = hostId;
        this.rawResponseError = rawResponseError;
    }

    /**
     * Gets error message.
     *
     * @return Error message.
     */
    public String getErrorMessage() {
        return errorMessage;
    }

    /**
     * Gets the error code.
     *
     * @return The error code in string.
     */
    public String getErrorCode() {
        return errorCode;
    }

    /**
     * Gets the request id.
     *
     * @return The request Id in string.
     */
    public String getRequestId() {
        return requestId;
    }

    /**
     * Gets the host id.
     *
     * @return The host Id in string.
     */
    public String getHostId() {
        return hostId;
    }

    /**
     * Gets the error message in OSS response.
     *
     * @return Error response in string.
     */
    public String getRawResponseError() {
        return rawResponseError;
    }

    /**
     * Sets the error response from OSS.
     *
     * @param rawResponseError The error response from OSS.
     */
    public void setRawResponseError(String rawResponseError) {
        this.rawResponseError = rawResponseError;
    }

    private String formatRawResponseError() {
        if (rawResponseError == null || rawResponseError.equals("")) {
            return "";
        }
        return String.format("\n[ResponseError]:\n%s", this.rawResponseError);
    }

    @Override
    public String getMessage() {
        return getErrorMessage() + "\n[ErrorCode]: " + getErrorCode() + "\n[RequestId]: " + getRequestId()
                + "\n[HostId]: " + getHostId() + formatRawResponseError();
    }
}
