package com.digiwin.dap.middleware.lmc.http.comm;

import com.digiwin.dap.middleware.lmc.common.auth.AuthCache;
import com.digiwin.dap.middleware.lmc.common.auth.BasicAuthCache;
import com.digiwin.dap.middleware.lmc.common.comm.IdleConnectionReaper;
import com.digiwin.dap.middleware.lmc.http.client.ClientConfiguration;
import com.digiwin.dap.middleware.lmc.http.utils.ExceptionFactory;
import com.digiwin.dap.middleware.lmc.http.utils.HttpHeaders;
import com.digiwin.dap.middleware.lmc.http.utils.HttpUtil;
import com.digiwin.dap.middleware.lmc.http.utils.IOUtils;
import com.digiwin.dap.middleware.lmc.internal.LmcConstant;
import org.apache.http.Header;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * Default implementation of {@link ServiceClient}.
 *
 * @author chenzhuang
 * @date 2021/4/23
 */
public class DefaultServiceClient extends ServiceClient {

    protected static HttpRequestFactory httpRequestFactory = new HttpRequestFactory();
    protected CloseableHttpClient httpClient;
    protected HttpClientConnectionManager connectionManager;
    protected RequestConfig requestConfig;
    protected AuthCache authCache = new BasicAuthCache();
    private volatile String endpoint = LmcConstant.DEFAULT_LMC_ENDPOINT;

    public DefaultServiceClient() {
        super(new ClientConfiguration());
        this.connectionManager = createHttpClientConnectionManager();
        this.httpClient = createHttpClient(this.connectionManager);

        RequestConfig.Builder requestConfigBuilder = RequestConfig.custom();
        requestConfigBuilder.setConnectTimeout(config.getConnectionTimeout());
        requestConfigBuilder.setSocketTimeout(config.getSocketTimeout());
        requestConfigBuilder.setConnectionRequestTimeout(config.getConnectionRequestTimeout());
        this.requestConfig = requestConfigBuilder.build();
    }

    @Override
    public ResponseMessage sendRequestCore(RequestMessage request) throws IOException {
        HttpRequestBase httpRequest = httpRequestFactory.createHttpRequest(request);
        HttpClientContext httpContext = createHttpContext();

        CloseableHttpResponse httpResponse;
        try {
            httpResponse = httpClient.execute(httpRequest, httpContext);
        } catch (IOException ex) {
            httpRequest.abort();
            throw ExceptionFactory.createNetworkException(ex);
        }

        return buildResponse(request, httpResponse);
    }

    private static ResponseMessage buildResponse(RequestMessage request, CloseableHttpResponse httpResponse)
            throws IOException {

        assert (httpResponse != null);

        ResponseMessage response = new ResponseMessage();
        response.setUrl(request.getEndpoint().toString());

        if (httpResponse.getStatusLine() != null) {
            response.setStatusCode(httpResponse.getStatusLine().getStatusCode());
        }

        if (httpResponse.getEntity() != null) {
            if (response.isSuccessful()) {
                response.setContent(httpResponse.getEntity().getContent());
            } else {
                readAndSetErrorResponse(httpResponse.getEntity().getContent(), response);
            }
        }

        for (Header header : httpResponse.getAllHeaders()) {
            if (HttpHeaders.CONTENT_LENGTH.equalsIgnoreCase(header.getName())) {
                response.setContentLength(Long.parseLong(header.getValue()));
            }
            response.addHeader(header.getName(), header.getValue());
        }

        HttpUtil.convertHeaderCharsetFromIso88591(response.getHeaders());
        return response;
    }

    private static void readAndSetErrorResponse(InputStream originalContent, ResponseMessage response)
            throws IOException {
        byte[] contentBytes = IOUtils.readStreamAsByteArray(originalContent);
        response.setErrorResponseAsString(new String(contentBytes));
        response.setContent(new ByteArrayInputStream(contentBytes));
    }

    private CloseableHttpClient createHttpClient(HttpClientConnectionManager connectionManager) {
        return HttpClients.custom().setConnectionManager(connectionManager)
                .disableContentCompression().disableAutomaticRetries().build();
    }

    private HttpClientConnectionManager createHttpClientConnectionManager() {
        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
        connectionManager.setDefaultMaxPerRoute(config.getMaxConnections());
        connectionManager.setMaxTotal(config.getMaxConnections());
        connectionManager.setDefaultSocketConfig(SocketConfig.custom().setSoTimeout(config.getSocketTimeout()).build());
        if (config.isUseReaper()) {
            IdleConnectionReaper.setIdleConnectionTime(config.getIdleConnectionTime());
            IdleConnectionReaper.registerConnectionManager(connectionManager);
        }
        return connectionManager;
    }

    private HttpClientContext createHttpContext() {
        HttpClientContext httpContext = HttpClientContext.create();
        httpContext.setRequestConfig(this.requestConfig);
        return httpContext;
    }

    @Override
    public void shutdown() {
        IdleConnectionReaper.removeConnectionManager(this.connectionManager);
        this.connectionManager.shutdown();
    }

    @Override
    public AuthCache getAuthCache() {
        return this.authCache;
    }

    @Override
    public void setAuthCache(String endpoint, String token) {
        this.endpoint = endpoint;
        this.authCache.put(endpoint, token);
    }



}
