package com.digiwin.dap.middleware.lmc.appender;

import ch.qos.logback.classic.spi.IThrowableProxy;
import ch.qos.logback.classic.spi.LoggingEvent;
import ch.qos.logback.classic.spi.StackTraceElementProxy;
import ch.qos.logback.classic.spi.ThrowableProxyUtil;
import ch.qos.logback.core.CoreConstants;
import ch.qos.logback.core.UnsynchronizedAppenderBase;
import ch.qos.logback.core.encoder.Encoder;
import com.digiwin.dap.middleware.lmc.http.client.HttpConfig;
import com.digiwin.dap.middleware.lmc.internal.LMCResourceUri;
import com.digiwin.dap.middleware.lmc.util.LogUtils;
import com.digiwin.dap.middleware.lmc.util.LoggingEventSizeUtil;
import com.digiwin.dap.middleware.lmc.util.ThreadPoolUtil;
import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.io.entity.StringEntity;
import org.apache.hc.core5.util.Timeout;
import org.json.JSONArray;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * LoghubAppender
 *
 * @author chenzhuang
 */
public class DwLogbackAppender<E> extends UnsynchronizedAppenderBase<E> {

    private SimpleDateFormat _sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

    private static final String CUSTOM_CONTENT_KEY_SEPARATOR = ",";

    private static List<String> CUSTOM_CONTENT_KEY_LIST = new ArrayList<>();

    /**
     * 默认间隔时间:2s
     */
    private static final int DEFAULT_PERIOD_MILL_SECONDS = 2;
    /**
     * 默认批量条数:50
     */
    private static final int DEFAULT_BATCH_SIZE = 50;
    /**
     * 默认单条日志最大大小:10KB
     */
    private static final Long DEFAULT_MAX_LOG_SIZE = 10240L;

    private static final Integer DEFAULT_THREAD_NUM = 5;

    private ScheduledExecutorService executor;

    private int consumeCount = 0;

    private Queue<Map<String, Object>> workQueue = new LinkedList<>();

    private CloseableHttpClient client;

    protected Encoder<E> encoder;
    private String userAgent = "logback";

    private Integer intervals;

    private Integer batchSize;

    private Long maxSingleLogSize;

    private String app;
    private String endpoint;

    private String customContentKey;


    @Override
    public void start() {
        if(null == this.intervals){
            this.intervals = DEFAULT_PERIOD_MILL_SECONDS;
        }
        if(null == this.batchSize){
            this.batchSize = DEFAULT_BATCH_SIZE;
        }
        if(null == this.maxSingleLogSize){
            this.maxSingleLogSize = DEFAULT_MAX_LOG_SIZE;
        }
        if(null != this.customContentKey && customContentKey.length() != 0){
            CUSTOM_CONTENT_KEY_LIST = Arrays.asList(this.customContentKey.split(CUSTOM_CONTENT_KEY_SEPARATOR));
        }
        // 请求配置
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(Timeout.ofMilliseconds(HttpConfig.CONNECT_TIMEOUT))
                .setConnectionRequestTimeout(Timeout.ofMilliseconds(HttpConfig.CONNECT_REQUEST_TIMEOUT))
                .setResponseTimeout(Timeout.ofMilliseconds(HttpConfig.SOCKET_TIMEOUT))
                .build();

        // 连接池管理
        PoolingHttpClientConnectionManager cm = new PoolingHttpClientConnectionManager();
        cm.setMaxTotal(HttpConfig.MAX_TOTAL_CONNECTIONS);
        cm.setDefaultMaxPerRoute(HttpConfig.MAXIMUM_CONNECTION_PER_ROUTE);
        cm.setValidateAfterInactivity(Timeout.ofMilliseconds(HttpConfig.CONNECTION_VALIDATE_AFTER_INACTIVITY_MS));

        this.client = HttpClients.custom()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionManager(cm).build();

        // 配置定时
        executor = Executors.newSingleThreadScheduledExecutor();
        executor.scheduleAtFixedRate(() -> {
            synchronized (DwLogbackAppender.this) {
                if (!workQueue.isEmpty()) {
                    List<Map<String, Object>> logList = new LinkedList<>();
                    for (Map<String, Object> element : workQueue) {
                        logList.add(element);
                    }
                    asyncPersistence(logList);
                    consumeCount = 0;
                    workQueue.clear();
                }
            }
        }, 0, this.intervals, TimeUnit.SECONDS);

        super.start();
    }

    @Override
    public void stop() {
        if (!isStarted()) {
            return;
        }
        List<Map<String, Object>> logList = new LinkedList<>();
        for (Map<String, Object> element : workQueue) {
            logList.add(element);
        }
        asyncPersistence(logList);
        executor.shutdown();
        super.stop();
        if (this.client != null) {
            try {
                this.client.close();
            } catch (IOException exception) {
                addError(exception.getMessage());
            }
        }
    }

    @Override
    public void append(E eventObject) {
        synchronized (this) {
            // 队列数据超出单次批量提交数量,说明当前生产者生产数据消费者未及时消费,阻塞线程,等待消费者消费
            while (this.workQueue.size() > this.batchSize) {
                try {
                    wait();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return;
                }
            }
            Map<String, Object> map = convertLoggingEventToMap(eventObject);
            if (Objects.nonNull(map)) {
                if (LoggingEventSizeUtil.getSize(map) >= this.maxSingleLogSize * 1024) {
                    // 单条日志超出1MB直接提交,不进入队列不计数且此处不清空队列,清空队列会导致未提交数据丢失
                    List<Map<String, Object>> log = Arrays.asList(map);
                    asyncPersistence(log);
                } else {
                    // 将新生产的数据放入队列,等待消费
                    this.workQueue.offer(map);
                    consumeCount++;
                }
            }

            // 满足消费条件,消费队列数据
            if (consumeCount >= this.batchSize) {
                List<Map<String, Object>> logList = new LinkedList<>();
                for (Map<String, Object> element : workQueue) {
                    logList.add(element);
                }
                asyncPersistence(logList);
                this.workQueue.clear();
                consumeCount = 0;
                // 队列数据被消费,唤醒等待的线程
                notifyAll();
            }
        }
    }

    private void asyncPersistence(List<Map<String, Object>> logList) {
        ThreadPoolUtil.executor(() -> {
            if (logList == null || logList.isEmpty()) {
                return;
            }

            HttpPost post = new HttpPost(LMCResourceUri.getSaveDevLogBatchUrl(this.endpoint));
            try {
                JSONArray jsonArray = new JSONArray(logList);
                post.setEntity(new StringEntity(jsonArray.toString(), ContentType.APPLICATION_JSON));

                ClassicHttpResponse response = this.client.execute(post, httpResponse -> httpResponse);

                if (response.getCode() != 200) {
                    post.cancel();
                }
            } catch (Exception e) {
                addError(e.getMessage());
            }
        });
    }

    private Map<String, Object> convertLoggingEventToMap(E eventObject) {
        if (!(eventObject instanceof LoggingEvent)) {
            return null;
        }
        LoggingEvent event = (LoggingEvent) eventObject;
        Map<String, Object> logMap = new HashMap();

        Map<String, String> mdcPropertyMap = event.getMDCPropertyMap();
        String traceId;
        if (!mdcPropertyMap.isEmpty()) {
            String ptxId = mdcPropertyMap.get("PtxId");
            //String pspanId = mdcPropertyMap.get("PspanId ");
            if (ptxId != null && !"".equals(ptxId)) {
                traceId = ptxId;
            } else {
                traceId = LogUtils.getUUID();
            }
            Map<String, String> customContent = new HashMap<>();
            CUSTOM_CONTENT_KEY_LIST.forEach(key -> {
                if (Objects.nonNull(mdcPropertyMap.get(key))) {
                    customContent.put(key, mdcPropertyMap.get(key));
                }
            });
            logMap.put("customContent", customContent);
        } else {
            traceId = LogUtils.getUUID();
        }
        logMap.putIfAbsent("traceId", traceId);

        Map<String, Object> initMap = new HashMap<>();
        LogUtils.initLogMap(initMap);
        logMap.putAll(initMap);
        logMap.putIfAbsent("appId", this.getApp());
        logMap.putIfAbsent("time", this._sdf.format(new Date(event.getTimeStamp())));
        logMap.putIfAbsent("level", event.getLevel().toString());
        logMap.putIfAbsent("thread", event.getThreadName());

        logMap.putIfAbsent("loggerName", event.getLoggerName());
        String addr = LogUtils.getLocalHostIpName();
        logMap.putIfAbsent("source", addr);

        logMap.putIfAbsent("appender", "DwLogbackAppender");

        StackTraceElement[] caller = event.getCallerData();
        if (caller != null && caller.length > 0) {
            logMap.putIfAbsent("location", caller[0].toString());
        }

        String message = event.getFormattedMessage();
        logMap.putIfAbsent("message", message);

        IThrowableProxy iThrowableProxy = event.getThrowableProxy();
        if (iThrowableProxy != null) {
            StringBuilder builder = new StringBuilder();

            String throwableProxyClassName = iThrowableProxy.getClassName();
            String throwableProxyMessage = iThrowableProxy.getMessage();

            String throwable = (throwableProxyMessage != null) ?
                    (throwableProxyClassName + ": " + throwableProxyMessage) : throwableProxyClassName;

            builder.append(throwable);
            for (StackTraceElementProxy step : event.getThrowableProxy().getStackTraceElementProxyArray()) {
                builder.append(CoreConstants.LINE_SEPARATOR);
                String string = step.toString();
                builder.append(CoreConstants.TAB).append(string);
                ThrowableProxyUtil.subjoinPackagingData(builder, step);
            }
            logMap.putIfAbsent("throwable", builder.toString());
        }

        if (this.encoder != null) {
            logMap.putIfAbsent("log", new String(this.encoder.encode(eventObject)));
        }
        return logMap;
    }



    public String getApp() {
        return app;
    }

    public void setApp(String app) {
        this.app = app;
    }

    public String getEndpoint() {
        return endpoint;
    }

    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    public Integer getIntervals() {
        return intervals;
    }

    public void setIntervals(Integer intervals) {
        this.intervals = intervals;
    }

    public Integer getBatchSize() {
        return batchSize;
    }

    public void setBatchSize(Integer batchSize) {
        this.batchSize = batchSize;
    }

    public Long getMaxSingleLogSize() {
        return maxSingleLogSize;
    }

    public void setMaxSingleLogSize(Long maxSingleLogSize) {
        this.maxSingleLogSize = maxSingleLogSize;
    }

    public String getCustomContentKey() {
        return customContentKey;
    }

    public void setCustomContentKey(String customContentKey) {
        this.customContentKey = customContentKey;
    }

    public Encoder<E> getEncoder() {
        return encoder;
    }

    public void setEncoder(Encoder<E> encoder) {
        this.encoder = encoder;
    }
}
