package com.digiwin.dap.nest.component.snowflake;

import com.alibaba.nacos.api.naming.pojo.Instance;
import com.digiwin.dap.nest.kernel.core.config.DwConfig;
import com.digiwin.dap.nest.kernel.core.config.JaEnvProperty;
import com.digiwin.dap.nest.kernel.core.config.DwProperty;
import com.digiwin.dap.nest.kernel.core.dapper.log.JaLog;
import com.digiwin.dap.nest.kernel.core.lock.distributed.JaDistributedLockAdapter;
import com.digiwin.dap.nest.kernel.core.lock.distributed.JaDistributedLockEntity;
import com.digiwin.dap.nest.kernel.core.util.DwStringUtil;
import com.digiwin.dap.nest.kernel.core.util.algorithm.JaMathUtil;
import com.digiwin.dap.nest.kernel.meta.exception.DwException;
import com.digiwin.dap.nest.infrastructure.middleware.nacos.config.meta.JaNacosConstants;
import com.digiwin.dap.nest.infrastructure.middleware.nacos.discovery.JaNacosDiscoveryPropertiesProcessor;
import com.digiwin.dap.nest.infrastructure.middleware.nacos.discovery.JaNamingService;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * 基于nacos顺序注册实现集群workId有序创建
 *
 * @author chenjian
 * @since 2025年11月07日 20:09:36
 */
@Configuration
public class JaSnowflakeConfiguration implements ApplicationListener<ApplicationReadyEvent> {
    @Override
    public void onApplicationEvent(@NonNull ApplicationReadyEvent event) {
        if (Boolean.TRUE.equals(DwConfig.getBoolean("id.snowflake.cluster.enabled", false))) {
            JaDistributedLockEntity lockEntity = JaDistributedLockEntity.builder()
                    .lockKey(String.format(DwConfig.KeyPrefix + "id.snowflake.clusterWorkId.init:%s", JaEnvProperty.getApplicationName()))
                    .spinTime(DwProperty.getLong(DwConfig.KeyPrefix + "id.snowflake.clusterWorkId.init.spinTime", 2L))
                    .spinNum(DwProperty.getLong(DwConfig.KeyPrefix + "id.snowflake.clusterWorkId.init.spinNum", 3L))
                    .build();
            JaSnowflakeClusterAdapter.setWorkerId(JaDistributedLockAdapter.lock(lockEntity, () -> {
                // 当前实例
                Instance currentInstance = JaNamingService.getCurrentInstanceWithRetry();
                if (currentInstance == null) {
                    throw new DwException("snowflake cluster workId init error, currentInstance is empty");
                }
                SortedSet<Long> idSet = collectUsedWorkerIds();
                // 最小可用id
                Long availableId = JaMathUtil.findMinAvailableId(idSet);

                currentInstance.getMetadata().put(JaNacosConstants.KeyMetaDataClusterId, availableId.toString());
                // 回写id
                JaNamingService.getNamingService().registerInstance(
                        JaNacosDiscoveryPropertiesProcessor.getNacosDiscoveryProperties().getService(),
                        JaNacosDiscoveryPropertiesProcessor.getNacosDiscoveryProperties().getGroup(),
                        currentInstance
                );
                return availableId;
            }));

            JaSnowflakeClusterAdapter.initClusterWorkerIdHandler();
            JaLog.info("snowflake cluster workId init : {}", JaSnowflakeClusterAdapter.getWorkerId());
        }
    }

    /**
     * 收集所有已使用的workerId
     */
    private SortedSet<Long> collectUsedWorkerIds() {
        // 当前实例List
        List<Instance> currentInstanceList = JaNamingService.getCurrentInstanceList();
        SortedSet<Long> idSet = new TreeSet<>();

        for (Instance instance : currentInstanceList) {
            // 只统计健康实例的ID
            if (!instance.isHealthy()) {
                JaLog.warn("skip unhealthy instance: {}", instance.getInstanceId());
                continue;
            }

            String clusterIdStr = instance.getMetadata().get(JaNacosConstants.KeyMetaDataClusterId);
            if (DwStringUtil.isEmpty(clusterIdStr)) {
                continue;
            }
            Long clusterId = Long.parseLong(clusterIdStr);

            // 检测ID重复
            if (!idSet.add(clusterId)) {
                throw new DwException(String.format(
                        "snowflake cluster workId init error: duplicate id=%d in instance=%s",
                        clusterId, instance.getInstanceId()));
            }
        }
        JaLog.info("snowflake workerId init: totalInstances={}, usedIds={}", currentInstanceList.size(), idSet);
        return idSet;
    }

}