package com.digiwin.dap.nest.component.snowflake;

import com.alibaba.nacos.api.naming.pojo.Instance;
import com.digiwin.dap.nest.infrastructure.middleware.nacos.config.meta.DwNacosConstants;
import com.digiwin.dap.nest.infrastructure.middleware.nacos.discovery.DwNacosDiscoveryPropertiesProcessor;
import com.digiwin.dap.nest.infrastructure.middleware.nacos.discovery.DwNamingService;
import com.digiwin.dap.nest.infrastructure.spring.util.DwSpringCloudFix;
import com.digiwin.dap.nest.kernel.core.config.DwConfig;
import com.digiwin.dap.nest.kernel.core.config.DwProperty;
import com.digiwin.dap.nest.kernel.core.config.DwEnvProperty;
import com.digiwin.dap.nest.kernel.core.dapper.log.DwLog;
import com.digiwin.dap.nest.kernel.core.lock.distributed.DwDistributedLockAdapter;
import com.digiwin.dap.nest.kernel.core.lock.distributed.DwDistributedLockEntity;
import com.digiwin.dap.nest.kernel.core.util.DwStringUtil;
import com.digiwin.dap.nest.kernel.core.util.algorithm.JaMathUtil;
import com.digiwin.dap.nest.kernel.meta.exception.DwException;
import org.springframework.cloud.client.discovery.event.InstanceRegisteredEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * 基于nacos顺序注册实现集群workId有序创建
 *
 * @author chenjian
 * @since 2025年11月07日 20:09:36
 */
@Order(0)
@Configuration
public class DwSnowflakeClusterConfiguration implements ApplicationListener<InstanceRegisteredEvent<?>> {
    @Override
    public void onApplicationEvent(@NonNull InstanceRegisteredEvent event) {
        if (!DwSpringCloudFix.isCloudContainer()) {
            if (Boolean.TRUE.equals(DwConfig.getBoolean("id.snowflake.cluster.enabled", false))) {
                DwDistributedLockEntity lockEntity = DwDistributedLockEntity.builder()
                        .lockKey(String.format(DwConfig.KeyPrefix + "id.snowflake.clusterWorkId.init:%s", DwEnvProperty.getApplicationName()))
                        .spinTime(DwProperty.getLong(DwConfig.KeyPrefix + "id.snowflake.clusterWorkId.init.spinTime", 2L))
                        .spinNum(DwProperty.getLong(DwConfig.KeyPrefix + "id.snowflake.clusterWorkId.init.spinNum", 3L))
                        .build();
                DwSnowflakeClusterAdapter.setWorkerId(DwDistributedLockAdapter.lock(lockEntity, () -> {
                    // 当前实例
                    Instance currentInstance = DwNamingService.getCurrentInstanceWithRetry();
                    if (currentInstance == null) {
                        throw new DwException("snowflake cluster workId init error, currentInstance is empty");
                    }
                    SortedSet<Long> idSet = collectUsedWorkerIds();
                    // 最小可用id
                    Long availableId = JaMathUtil.findMinAvailableId(idSet);

                    currentInstance.getMetadata().put(DwNacosConstants.KeyMetaDataClusterId, availableId.toString());
                    // 回写id
                    DwNamingService.getNamingService().registerInstance(
                            DwNacosDiscoveryPropertiesProcessor.getNacosDiscoveryProperties().getService(),
                            DwNacosDiscoveryPropertiesProcessor.getNacosDiscoveryProperties().getGroup(),
                            currentInstance
                    );
                    return availableId;
                }));

                DwSnowflakeClusterAdapter.initClusterWorkerIdHandler();
                DwLog.info("snowflake cluster workId init : {}", DwSnowflakeClusterAdapter.getWorkerId());
            }
        }

    }

    /**
     * 收集所有已使用的workerId
     */
    private SortedSet<Long> collectUsedWorkerIds() {
        // 当前实例List
        List<Instance> currentInstanceList = DwNamingService.getCurrentInstanceList();
        SortedSet<Long> idSet = new TreeSet<>();

        for (Instance instance : currentInstanceList) {
            // 只统计健康实例的ID
            if (!instance.isHealthy()) {
                DwLog.warn("skip unhealthy instance: {}", instance.getInstanceId());
                continue;
            }

            String clusterIdStr = instance.getMetadata().get(DwNacosConstants.KeyMetaDataClusterId);
            if (DwStringUtil.isEmpty(clusterIdStr)) {
                continue;
            }
            Long clusterId = Long.parseLong(clusterIdStr);

            // 检测ID重复
            if (!idSet.add(clusterId)) {
                throw new DwException(String.format(
                        "snowflake cluster workId init error: duplicate id=%d in instance=%s",
                        clusterId, instance.getInstanceId()));
            }
        }
        DwLog.info("snowflake workerId init: totalInstances={}, usedIds={}", currentInstanceList.size(), idSet);
        return idSet;
    }

}