package com.digiwin.dap.nest.infrastructure.middleware.nacos.config;

import com.alibaba.nacos.api.NacosFactory;
import com.alibaba.nacos.api.PropertyKeyConst;
import com.alibaba.nacos.api.config.ConfigService;
import com.digiwin.dap.nest.kernel.core.config.DwEnvProperty;
import com.digiwin.dap.nest.kernel.core.config.DwProperty;
import com.digiwin.dap.nest.kernel.core.config.processor.DwYamlProcessor;
import com.digiwin.dap.nest.kernel.core.dapper.log.DwLog;
import com.digiwin.dap.nest.kernel.core.util.DwShutdownHookUtil;
import com.digiwin.dap.nest.kernel.core.util.DwStringUtil;
import com.digiwin.dap.nest.kernel.core.util.DwThrowableUtil;
import com.digiwin.dap.nest.kernel.core.util.DwValidateUtil;
import com.digiwin.dap.nest.kernel.core.util.datastructure.DwCollectionUtil;
import com.digiwin.dap.nest.kernel.core.util.reflect.clazz.JaClassUtil;
import com.digiwin.dap.nest.infrastructure.middleware.nacos.config.listenter.DwNacosConfigListener;
import com.digiwin.dap.nest.infrastructure.middleware.nacos.config.meta.DwNacosConfigEntity;
import com.digiwin.dap.nest.infrastructure.middleware.nacos.config.meta.DwNacosConfigServerEntity;
import lombok.SneakyThrows;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * 读取nacos配置
 *
 * @author chenjian
 * @since 2020年07月13日 12:05:52
 */
public class DwNacosConfigService {

    static {
        // com.alibaba.nacos.client.logging.AbstractNacosLogging.getLocation
        if (JaClassUtil.hashClass("ch.qos.logback.classic.LoggerContext") && DwLog.getUseLogback()) {
            System.setProperty("nacos.logging.config", "classpath:jugg-nacos-logback.xml");
        }
    }

    /**
     * 读取nacos配置
     *
     * @author chenjian
     * @since 2020年07月29日 12:06:32
     */
    @SneakyThrows
    public static String read(DwNacosConfigEntity entity, Properties properties) {
        if (!DwNacosConfigPropertiesProcessor.getEnabled()) {
            return null;
        }
        String dataId = entity.getDataId();
        DwValidateUtil.notNull(dataId, () -> "read nacos dataId is null");
        String group = entity.getGroup();
        DwValidateUtil.notNull(group, () -> "read nacos group is null");
        String namespace = entity.getNamespace();
        DwValidateUtil.notNull(namespace, () -> "read nacos namespace is null");
        DwValidateUtil.notNull(properties, () -> "read nacos properties is null");
        properties.put(PropertyKeyConst.NAMESPACE, namespace);

        ConfigService configService = NacosFactory.createConfigService(properties);
        String content = configService.getConfig(dataId, entity.getGroup(), entity.getTimeoutMs());
        if (null != entity.getListener()) {
            configService.addListener(dataId, entity.getGroup(), entity.getListener());
            DwShutdownHookUtil.add(dataId, () -> DwThrowableUtil.tryCatchRun(configService::shutDown));
        } else {
            configService.shutDown();
        }
        DwLog.info("{} {} nacos config : \n{}", dataId, group, content);
        return content;
    }

    public static String read(DwNacosConfigEntity entity, DwNacosConfigServerEntity serverEntity) {
        return read(entity, DwNacosConfigPropertiesProcessor.getServerAddrProperties(serverEntity));
    }

    public static Map<String, Object> readYaml(DwNacosConfigEntity entity, Properties properties) {
        String content = read(entity, properties);
        return DwStringUtil.isEmpty(content) ? new HashMap<>(0) : DwYamlProcessor.getFlattenedMap(content);
    }

    public static Map<String, Object> readYaml(DwNacosConfigEntity entity, DwNacosConfigServerEntity serverEntity) {
        return readYaml(entity, DwNacosConfigPropertiesProcessor.getServerAddrProperties(serverEntity));
    }

    public static Map<String, Object> readYaml(DwNacosConfigEntity entity) {
        Properties serverAddrProperties;
        try {
            serverAddrProperties = DwNacosConfigPropertiesProcessor.getServerAddrProperties();
            if (null == serverAddrProperties) {
                DwLog.warn("{} nacos readYaml error, serverAddrProperties is null", entity.getDataId());
                return null;
            }
        } catch (NullPointerException e) {
            DwLog.warn("{} nacos readYaml error, serverAddrProperties is null", entity.getDataId());
            return null;
        }
        return readYaml(entity, serverAddrProperties);
    }

    public static Map<String, Object> readYaml(String dataId, String namespace, DwNacosConfigListener listener) {
        return DwNacosConfigService.readYaml(DwNacosConfigEntity.builder()
                .dataId(dataId)
                .namespace(namespace)
                .group(DwEnvProperty.getVersion())
                .listener(listener)
                .build());
    }

    public static Map<String, Object> readYaml(String dataId, String namespace) {
        return readYaml(dataId, namespace, null);
    }

    public static void loadYaml(String dataId, String namespace) {
        Map<String, Object> nacosMap = readYaml(dataId, namespace, null);
        if (DwCollectionUtil.isNotEmpty(nacosMap)) {
            DwProperty.getPropertyMap().putAll(nacosMap);
        }
    }


    public static void main(String[] args) throws IOException {
        DwNacosConfigListener jaNacosConfigListener = new DwNacosConfigListener() {

            @Override
            public void receiveConfigInfo(String configInfo) {
                System.out.println("configInfo : " + configInfo);
            }
        };
        Map<String, Object> read = DwNacosConfigService.readYaml(DwNacosConfigEntity.builder()
                        .namespace("8231cc09-273b-43db-8561-6a82f21f15e0")
                        .dataId("rabbitmq.yml")
                        .group("2.2.21")
                        .listener(jaNacosConfigListener)
                        .build()
                , DwNacosConfigServerEntity.builder()
                        .serverAddr("http://121.37.176.231:8848")
                        .username("")
                        .password("")
                        .build()
        );
        System.out.println(read);
        System.in.read();
    }
}
