package com.digiwin.dap.nest.infrastructure.middleware.nacos.discovery;

import com.alibaba.cloud.nacos.NacosDiscoveryProperties;
import com.alibaba.cloud.nacos.NacosServiceManager;
import com.alibaba.cloud.nacos.registry.NacosServiceRegistry;
import com.alibaba.nacos.api.naming.NamingService;
import com.alibaba.nacos.api.naming.pojo.Instance;
import com.digiwin.dap.nest.kernel.core.config.DwConfig;
import com.digiwin.dap.nest.kernel.core.dapper.log.DwLog;
import com.digiwin.dap.nest.kernel.core.util.DwShutdownHookUtil;
import com.digiwin.dap.nest.kernel.core.util.DwStringUtil;
import com.digiwin.dap.nest.kernel.core.util.datastructure.DwCollectionUtil;
import com.digiwin.dap.nest.infrastructure.spring.util.DwSpringBeanUtil;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Objects;

/**
 * 应对一个服务注册多个服务名, 慎用, 建议使用starter原生功能
 * TODO 未实现 Watch subscribe unsubscribe
 *
 * @author chenjian
 * @since 2025年03月21日 21:08:00
 */
@Slf4j
public class DwNamingService {
    private DwNamingService() {
    }

    private static NamingService namingService;

    public static NamingService getNamingService() {
        return namingService;
    }

    public static void setNamingService(NamingService namingService) {
        DwNamingService.namingService = namingService;
    }

    /**
     * 应对一个服务注册多个服务名, 慎用, 建议使用starter原生功能
     * spring.cloud.nacos.discovery.service
     * TODO 未实现 Watch subscribe unsubscribe
     */
    @SneakyThrows
    public static void registerInstancesByCurrentInstance() {
        String services = DwConfig.get("nacos.discovery.services");
        if (DwStringUtil.isEmpty(services)) {
            return;
        }
        String[] serviceArray = services.split(",");
        if (DwCollectionUtil.isEmpty(serviceArray)) {
            return;
        }
        NamingService namingService = getNamingServiceByNacosServiceManager();
        if (null == namingService) {
            DwLog.warn("registerInstancesByCurrentInstance : namingService is null");
            return;
        }
        for (String service : serviceArray) {
            registerInstance(namingService, DwNacosDiscoveryPropertiesProcessor.createByCurrent(service.trim()));
        }
    }


    /**
     * @see NacosServiceRegistry#register(org.springframework.cloud.client.serviceregistry.Registration)
     * @see NacosServiceRegistry#deregister(org.springframework.cloud.client.serviceregistry.Registration)
     */
    @SneakyThrows
    public static void registerInstance(NamingService namingService, NacosDiscoveryProperties nacosDiscoveryProperties) {
//        NamingService namingService = NamingFactory.createNamingService(nacosDiscoveryProperties.getNacosProperties());
        namingService.registerInstance(nacosDiscoveryProperties.getService()
                , nacosDiscoveryProperties.getGroup()
                , getNacosInstanceFromRegistration(nacosDiscoveryProperties));
        DwShutdownHookUtil.add("nacos namingService deregisterInstance:" + nacosDiscoveryProperties.getService(), () -> {
            try {
                namingService.deregisterInstance(nacosDiscoveryProperties.getService()
                        , nacosDiscoveryProperties.getGroup()
                        , nacosDiscoveryProperties.getIp()
                        , nacosDiscoveryProperties.getPort()
                        , nacosDiscoveryProperties.getClusterName());
            } catch (Throwable e) {
                log.error("ERR_NACOS_DEREGISTER, de-register failed...{},", nacosDiscoveryProperties.toString(), e);
            }
        });
    }

    /**
     * 通过 NacosServiceManager 获取 NamingService
     *
     * @see NacosServiceManager#getNamingService(java.util.Properties) 获取 NamingService
     */
    public static NamingService getNamingServiceByNacosServiceManager() {
        NacosServiceManager bean = DwSpringBeanUtil.getBean(NacosServiceManager.class);
        if (null == bean) {
            return null;
        }
        return bean.getNamingService();
    }

    public static void deregisterCurrentInstance() {
        NacosDiscoveryProperties nacosDiscoveryProperties = DwSpringBeanUtil.getBean(NacosDiscoveryProperties.class);
        assert nacosDiscoveryProperties != null;
        DwShutdownHookUtil.add("nacos namingService deregisterInstance:" + nacosDiscoveryProperties.getService(), () -> {
            try {
                Objects.requireNonNull(DwNamingService.getNamingServiceByNacosServiceManager()).deregisterInstance(nacosDiscoveryProperties.getService()
                        , nacosDiscoveryProperties.getGroup()
                        , nacosDiscoveryProperties.getIp()
                        , nacosDiscoveryProperties.getPort()
                        , nacosDiscoveryProperties.getClusterName());
            } catch (Throwable e) {
                DwLog.error("ERR_NACOS_DEREGISTER, de-register failed...{},", nacosDiscoveryProperties.toString(), e);
            }
        });
    }

    /**
     * @see NacosServiceRegistry#getNacosInstanceFromRegistration(org.springframework.cloud.client.serviceregistry.Registration)
     */
    private static Instance getNacosInstanceFromRegistration(NacosDiscoveryProperties nacosDiscoveryProperties) {
        Instance instance = new Instance();
        instance.setIp(nacosDiscoveryProperties.getIp());
        instance.setPort(nacosDiscoveryProperties.getPort());
        instance.setWeight(nacosDiscoveryProperties.getWeight());
        instance.setClusterName(nacosDiscoveryProperties.getClusterName());
        instance.setEnabled(nacosDiscoveryProperties.isInstanceEnabled());
        instance.setMetadata(nacosDiscoveryProperties.getMetadata());
        instance.setEphemeral(nacosDiscoveryProperties.isEphemeral());
        return instance;
    }

    /**
     * 获取当前服务实例List
     * Nacos 注册之后, 再获取实例有延迟, 不一定能获取到最新
     * Nacos 客户端缓存刷新周期：默认是 5 秒
     *
     * @see com.alibaba.nacos.client.naming.core.HostReactor
     */
    @SneakyThrows
    public static List<Instance> getCurrentInstanceList() {
        NacosDiscoveryProperties nacosDiscoveryProperties = DwNacosDiscoveryPropertiesProcessor.getNacosDiscoveryProperties();
        return getNamingService().getAllInstances(nacosDiscoveryProperties.getService(), nacosDiscoveryProperties.getGroup());
    }

    /**
     * 获取当前服务实例
     * nacos注册之后, 再获取实例有延迟, 不一定能获取到最新
     * Nacos 客户端缓存刷新周期：默认是 5 秒
     *
     * @see com.alibaba.nacos.client.naming.core.HostReactor
     */
    public static Instance getCurrentInstance() {
        NacosDiscoveryProperties nacosDiscoveryProperties = DwNacosDiscoveryPropertiesProcessor.getNacosDiscoveryProperties();
        String ip = nacosDiscoveryProperties.getIp();
        int port = nacosDiscoveryProperties.getPort();
        List<Instance> instanceList = DwNamingService.getCurrentInstanceList();
        if (DwCollectionUtil.isNotEmpty(instanceList)) {
            for (Instance instance : instanceList) {
                if (instance.getIp().equals(ip) && port == instance.getPort()) {
                    return instance;
                }
            }
        }

        return null;
    }

    /**
     * 重试多次获取当前服务实例
     */
    public static Instance getCurrentInstanceWithRetry() {
        int retryCount = 0;
        int maxCount = 5;
        // 初始等待时间为2秒（可根据需求调整）
        // 当前轮次的延迟时间
        long currentDelay = 2000L;

        while (retryCount < maxCount) {
            try {
                Instance instance = DwNamingService.getCurrentInstance();
                if (instance != null) {
                    return instance;
                }
                log.info("获取Nacos服务实例重试, 重试次数: {}, 本次等待时间: {}ms", retryCount, currentDelay);
                Thread.sleep(currentDelay); // 使用动态计算的等待时间
            } catch (Exception e) {
                log.error("获取Nacos服务实例失败，重试次数: {}, 本次等待时间: {}ms", retryCount, currentDelay, e);
                try {
                    Thread.sleep(currentDelay); // 异常时同样等待相同时间
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException("线程被中断", ie); // 显式抛出中断异常
                }
            } finally {
                retryCount++;
                // 更新下一次的等待时间（指数退避策略）
                if (retryCount < maxCount) {
                    currentDelay = Math.min(currentDelay * 2, 30000);
                }
            }
        }
        return null;
    }

}
